/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.core;

import com.google.bitcoin.core.Block;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.ScriptException;
import com.google.bitcoin.core.Sha256Hash;
import com.google.bitcoin.core.StoredBlock;
import com.google.bitcoin.core.Transaction;
import com.google.bitcoin.core.Utils;
import com.google.bitcoin.core.VerificationException;
import com.google.bitcoin.core.Wallet;
import com.google.bitcoin.store.BlockStore;
import com.google.bitcoin.store.BlockStoreException;
import com.google.common.base.Preconditions;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class BlockChain {
    private static final Logger log = LoggerFactory.getLogger(BlockChain.class);
    protected final BlockStore blockStore;
    protected StoredBlock chainHead;
    protected final Object chainHeadLock = new Object();
    protected final NetworkParameters params;
    protected final List<Wallet> wallets;
    private final LinkedHashMap<Sha256Hash, Block> orphanBlocks = new LinkedHashMap();
    private long statsLastTime = System.currentTimeMillis();
    private long statsBlocksAdded;
    private static Date testnetDiffDate = new Date(1329264000000L);

    public BlockChain(NetworkParameters params, Wallet wallet, BlockStore blockStore) throws BlockStoreException {
        this(params, new ArrayList<Wallet>(), blockStore);
        if (wallet != null) {
            this.addWallet(wallet);
        }
    }

    public BlockChain(NetworkParameters params, BlockStore blockStore) throws BlockStoreException {
        this(params, new ArrayList<Wallet>(), blockStore);
    }

    public BlockChain(NetworkParameters params, List<Wallet> wallets, BlockStore blockStore) throws BlockStoreException {
        this.blockStore = blockStore;
        this.chainHead = blockStore.getChainHead();
        log.info("chain head is at height {}:\n{}", this.chainHead.getHeight(), (Object)this.chainHead.getHeader());
        this.params = params;
        this.wallets = new ArrayList<Wallet>(wallets);
    }

    public synchronized void addWallet(Wallet wallet) {
        this.wallets.add(wallet);
    }

    public BlockStore getBlockStore() {
        return this.blockStore;
    }

    public synchronized boolean add(Block block) throws VerificationException, ScriptException {
        try {
            return this.add(block, true);
        }
        catch (BlockStoreException e) {
            throw new RuntimeException(e);
        }
    }

    private synchronized boolean add(Block block, boolean tryConnecting) throws BlockStoreException, VerificationException, ScriptException {
        if (System.currentTimeMillis() - this.statsLastTime > 1000L) {
            if (this.statsBlocksAdded > 1L) {
                log.info("{} blocks per second", this.statsBlocksAdded);
            }
            this.statsLastTime = System.currentTimeMillis();
            this.statsBlocksAdded = 0L;
        }
        if (block.equals(this.getChainHead().getHeader()) || tryConnecting && this.orphanBlocks.containsKey(block.getHash())) {
            return true;
        }
        boolean contentsImportant = false;
        if (block.transactions != null) {
            contentsImportant = this.containsRelevantTransactions(block);
        }
        try {
            block.verifyHeader();
            if (contentsImportant) {
                block.verifyTransactions();
            }
        }
        catch (VerificationException e) {
            log.error("Failed to verify block: ", e);
            log.error(block.getHashAsString());
            throw e;
        }
        StoredBlock storedPrev = this.blockStore.get(block.getPrevBlockHash());
        if (storedPrev == null) {
            Preconditions.checkState(tryConnecting, "bug in tryConnectingOrphans");
            log.warn("Block does not connect: {} prev {}", (Object)block.getHashAsString(), (Object)block.getPrevBlockHash());
            this.orphanBlocks.put(block.getHash(), block);
            return false;
        }
        StoredBlock newStoredBlock = storedPrev.build(block);
        this.checkDifficultyTransitions(storedPrev, newStoredBlock);
        this.blockStore.put(newStoredBlock);
        this.connectBlock(newStoredBlock, storedPrev, block.transactions);
        if (tryConnecting) {
            this.tryConnectingOrphans();
        }
        ++this.statsBlocksAdded;
        return true;
    }

    private void connectBlock(StoredBlock newStoredBlock, StoredBlock storedPrev, List<Transaction> transactions) throws BlockStoreException, VerificationException {
        StoredBlock head = this.getChainHead();
        if (storedPrev.equals(head)) {
            this.setChainHead(newStoredBlock);
            log.debug("Chain is now {} blocks high", newStoredBlock.getHeight());
            for (Wallet wallet : this.wallets) {
                if (transactions != null) {
                    this.sendTransactionsToWallet(newStoredBlock, NewBlockType.BEST_CHAIN, wallet, transactions);
                }
                wallet.notifyNewBestBlock(newStoredBlock.getHeader());
            }
        } else {
            boolean haveNewBestChain = newStoredBlock.moreWorkThan(head);
            if (haveNewBestChain) {
                log.info("Block is causing a re-organize");
            } else {
                StoredBlock splitPoint = this.findSplit(newStoredBlock, head);
                if (splitPoint == newStoredBlock) {
                    log.warn("Saw duplicated block in main chain at height {}: {}", newStoredBlock.getHeight(), (Object)newStoredBlock.getHeader().getHash());
                    return;
                }
                if (splitPoint == null) {
                    log.error("Block forks the chain but splitPoint is null");
                } else {
                    int splitPointHeight = splitPoint.getHeight();
                    String splitPointHash = splitPoint.getHeader().getHashAsString();
                    log.info("Block forks the chain at height {}/block {}, but it did not cause a reorganize:\n{}", new Object[]{splitPointHeight, splitPointHash, newStoredBlock});
                }
            }
            if (transactions != null) {
                for (Wallet wallet : this.wallets) {
                    this.sendTransactionsToWallet(newStoredBlock, NewBlockType.SIDE_CHAIN, wallet, transactions);
                }
            }
            if (haveNewBestChain) {
                this.handleNewBestChain(newStoredBlock);
            }
        }
    }

    private void handleNewBestChain(StoredBlock newChainHead) throws BlockStoreException, VerificationException {
        StoredBlock head = this.getChainHead();
        StoredBlock splitPoint = this.findSplit(newChainHead, head);
        log.info("Re-organize after split at height {}", splitPoint.getHeight());
        log.info("Old chain head: {}", (Object)head.getHeader().getHashAsString());
        log.info("New chain head: {}", (Object)newChainHead.getHeader().getHashAsString());
        log.info("Split at block: {}", (Object)splitPoint.getHeader().getHashAsString());
        List<StoredBlock> oldBlocks = this.getPartialChain(head, splitPoint);
        List<StoredBlock> newBlocks = this.getPartialChain(newChainHead, splitPoint);
        for (Wallet wallet : this.wallets) {
            wallet.reorganize(splitPoint, oldBlocks, newBlocks);
        }
        this.setChainHead(newChainHead);
    }

    private List<StoredBlock> getPartialChain(StoredBlock higher, StoredBlock lower) throws BlockStoreException {
        Preconditions.checkArgument(higher.getHeight() > lower.getHeight(), "higher and lower are reversed");
        LinkedList<StoredBlock> results = new LinkedList<StoredBlock>();
        StoredBlock cursor = higher;
        do {
            results.add(cursor);
        } while (!(cursor = Preconditions.checkNotNull(cursor.getPrev(this.blockStore), "Ran off the end of the chain")).equals(lower));
        return results;
    }

    private StoredBlock findSplit(StoredBlock newChainHead, StoredBlock oldChainHead) throws BlockStoreException {
        StoredBlock currentChainCursor = oldChainHead;
        StoredBlock newChainCursor = newChainHead;
        while (!currentChainCursor.equals(newChainCursor)) {
            if (currentChainCursor.getHeight() > newChainCursor.getHeight()) {
                currentChainCursor = currentChainCursor.getPrev(this.blockStore);
                Preconditions.checkNotNull(currentChainCursor, "Attempt to follow an orphan chain");
                continue;
            }
            newChainCursor = newChainCursor.getPrev(this.blockStore);
            Preconditions.checkNotNull(newChainCursor, "Attempt to follow an orphan chain");
        }
        return currentChainCursor;
    }

    public int getBestChainHeight() {
        return this.getChainHead().getHeight();
    }

    private void sendTransactionsToWallet(StoredBlock block, NewBlockType blockType, Wallet wallet, List<Transaction> transactions) throws VerificationException {
        for (Transaction tx : transactions) {
            try {
                if (!wallet.isTransactionRelevant(tx, true)) continue;
                wallet.receiveFromBlock(tx, block, blockType);
            }
            catch (ScriptException e) {
                log.warn("Failed to parse a script: " + e.toString());
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void setChainHead(StoredBlock chainHead) throws BlockStoreException {
        this.blockStore.setChainHead(chainHead);
        Object object = this.chainHeadLock;
        synchronized (object) {
            this.chainHead = chainHead;
        }
    }

    private void tryConnectingOrphans() throws VerificationException, ScriptException, BlockStoreException {
        int blocksConnectedThisRound;
        do {
            blocksConnectedThisRound = 0;
            Iterator<Block> iter = this.orphanBlocks.values().iterator();
            while (iter.hasNext()) {
                Block block = iter.next();
                log.debug("Trying to connect {}", block.getHash());
                StoredBlock prev = this.blockStore.get(block.getPrevBlockHash());
                if (prev == null) {
                    log.debug("  but it is not connectable right now");
                    continue;
                }
                this.add(block, false);
                iter.remove();
                ++blocksConnectedThisRound;
            }
            if (blocksConnectedThisRound <= 0) continue;
            log.info("Connected {} orphan blocks.", blocksConnectedThisRound);
        } while (blocksConnectedThisRound > 0);
    }

    private void checkDifficultyTransitions(StoredBlock storedPrev, StoredBlock storedNext) throws BlockStoreException, VerificationException {
        Block prev = storedPrev.getHeader();
        Block next = storedNext.getHeader();
        if ((storedPrev.getHeight() + 1) % this.params.interval != 0) {
            if (this.params.getId().equals("org.bitcoin.test") && next.getTime().after(testnetDiffDate)) {
                this.checkTestnetDifficulty(storedPrev, prev, next);
                return;
            }
            if (next.getDifficultyTarget() != prev.getDifficultyTarget()) {
                throw new VerificationException("Unexpected change in difficulty at height " + storedPrev.getHeight() + ": " + Long.toHexString(next.getDifficultyTarget()) + " vs " + Long.toHexString(prev.getDifficultyTarget()));
            }
            return;
        }
        long now = System.currentTimeMillis();
        StoredBlock cursor = this.blockStore.get(prev.getHash());
        for (int i = 0; i < this.params.interval - 1; ++i) {
            if (cursor == null) {
                throw new VerificationException("Difficulty transition point but we did not find a way back to the genesis block.");
            }
            cursor = this.blockStore.get(cursor.getHeader().getPrevBlockHash());
        }
        log.info("Difficulty transition traversal took {}msec", System.currentTimeMillis() - now);
        Block blockIntervalAgo = cursor.getHeader();
        int timespan = (int)(prev.getTimeSeconds() - blockIntervalAgo.getTimeSeconds());
        if (timespan < this.params.targetTimespan / 4) {
            timespan = this.params.targetTimespan / 4;
        }
        if (timespan > this.params.targetTimespan * 4) {
            timespan = this.params.targetTimespan * 4;
        }
        BigInteger newDifficulty = Utils.decodeCompactBits(prev.getDifficultyTarget());
        newDifficulty = newDifficulty.multiply(BigInteger.valueOf(timespan));
        if ((newDifficulty = newDifficulty.divide(BigInteger.valueOf(this.params.targetTimespan))).compareTo(this.params.proofOfWorkLimit) > 0) {
            log.info("Difficulty hit proof of work limit: {}", (Object)newDifficulty.toString(16));
            newDifficulty = this.params.proofOfWorkLimit;
        }
        int accuracyBytes = (int)(next.getDifficultyTarget() >>> 24) - 3;
        BigInteger receivedDifficulty = next.getDifficultyTargetAsInteger();
        BigInteger mask = BigInteger.valueOf(0xFFFFFFL).shiftLeft(accuracyBytes * 8);
        if ((newDifficulty = newDifficulty.and(mask)).compareTo(receivedDifficulty) != 0) {
            throw new VerificationException("Network provided difficulty bits do not match what was calculated: " + receivedDifficulty.toString(16) + " vs " + newDifficulty.toString(16));
        }
    }

    private void checkTestnetDifficulty(StoredBlock storedPrev, Block prev, Block next) throws VerificationException, BlockStoreException {
        long timeDelta = next.getTimeSeconds() - prev.getTimeSeconds();
        if (timeDelta >= 0L && timeDelta <= 1200L) {
            BigInteger newDifficulty;
            StoredBlock cursor = storedPrev;
            while (!cursor.getHeader().equals(this.params.genesisBlock) && cursor.getHeight() % this.params.interval != 0 && cursor.getHeader().getDifficultyTargetAsInteger().equals(this.params.proofOfWorkLimit)) {
                cursor = cursor.getPrev(this.blockStore);
            }
            BigInteger cursorDifficulty = cursor.getHeader().getDifficultyTargetAsInteger();
            if (!cursorDifficulty.equals(newDifficulty = next.getDifficultyTargetAsInteger())) {
                throw new VerificationException("Testnet block transition that is not allowed: " + Long.toHexString(cursor.getHeader().getDifficultyTarget()) + " vs " + Long.toHexString(next.getDifficultyTarget()));
            }
        }
    }

    private boolean containsRelevantTransactions(Block block) {
        for (Transaction tx : block.transactions) {
            try {
                for (Wallet wallet : this.wallets) {
                    if (!wallet.isTransactionRelevant(tx, true)) continue;
                    return true;
                }
            }
            catch (ScriptException e) {
                log.warn("Failed to parse a script: " + e.toString());
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public StoredBlock getChainHead() {
        Object object = this.chainHeadLock;
        synchronized (object) {
            return this.chainHead;
        }
    }

    public synchronized Block getOrphanRoot(Sha256Hash from) {
        Block tmp;
        Block cursor = this.orphanBlocks.get(from);
        if (cursor == null) {
            return null;
        }
        while ((tmp = this.orphanBlocks.get(cursor.getPrevBlockHash())) != null) {
            cursor = tmp;
        }
        return cursor;
    }

    public synchronized boolean isOrphan(Sha256Hash block) {
        return this.orphanBlocks.containsKey(block);
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    public static enum NewBlockType {
        BEST_CHAIN,
        SIDE_CHAIN;

    }
}

