/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.core;

import com.google.bitcoin.core.Address;
import com.google.bitcoin.core.DumpedPrivateKey;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.UnsafeByteArrayOutputStream;
import com.google.bitcoin.core.Utils;
import com.google.common.base.Preconditions;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.Serializable;
import java.math.BigInteger;
import java.security.SecureRandom;
import java.util.Arrays;
import org.spongycastle.asn1.ASN1Encodable;
import org.spongycastle.asn1.ASN1InputStream;
import org.spongycastle.asn1.ASN1Integer;
import org.spongycastle.asn1.ASN1OctetString;
import org.spongycastle.asn1.DERBitString;
import org.spongycastle.asn1.DERInteger;
import org.spongycastle.asn1.DEROctetString;
import org.spongycastle.asn1.DERSequenceGenerator;
import org.spongycastle.asn1.DERTaggedObject;
import org.spongycastle.asn1.DLSequence;
import org.spongycastle.asn1.sec.SECNamedCurves;
import org.spongycastle.asn1.x9.X9ECParameters;
import org.spongycastle.crypto.AsymmetricCipherKeyPair;
import org.spongycastle.crypto.generators.ECKeyPairGenerator;
import org.spongycastle.crypto.params.ECDomainParameters;
import org.spongycastle.crypto.params.ECKeyGenerationParameters;
import org.spongycastle.crypto.params.ECPrivateKeyParameters;
import org.spongycastle.crypto.params.ECPublicKeyParameters;
import org.spongycastle.crypto.signers.ECDSASigner;

public class ECKey
implements Serializable {
    private static final ECDomainParameters ecParams;
    private static final SecureRandom secureRandom;
    private static final long serialVersionUID = -728224901792295832L;
    public static final int PUBLIC_KEY_LENGTH = 65;
    private BigInteger priv;
    private byte[] pub;
    private long creationTimeSeconds;
    private transient byte[] pubKeyHash;

    public ECKey() {
        ECKeyPairGenerator generator = new ECKeyPairGenerator();
        ECKeyGenerationParameters keygenParams = new ECKeyGenerationParameters(ecParams, secureRandom);
        generator.init(keygenParams);
        AsymmetricCipherKeyPair keypair = generator.generateKeyPair();
        ECPrivateKeyParameters privParams = (ECPrivateKeyParameters)keypair.getPrivate();
        ECPublicKeyParameters pubParams = (ECPublicKeyParameters)keypair.getPublic();
        this.priv = privParams.getD();
        this.pub = pubParams.getQ().getEncoded();
        this.creationTimeSeconds = Utils.now().getTime() / 1000L;
    }

    public static ECKey fromASN1(byte[] asn1privkey) {
        return new ECKey(ECKey.extractPrivateKeyFromASN1(asn1privkey));
    }

    public byte[] toASN1() {
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream(400);
            DERSequenceGenerator seq = new DERSequenceGenerator(baos);
            seq.addObject(new ASN1Integer(1));
            seq.addObject(new DEROctetString(this.priv.toByteArray()));
            seq.addObject(new DERTaggedObject(0, SECNamedCurves.getByName("secp256k1").toASN1Primitive()));
            seq.addObject(new DERTaggedObject(1, new DERBitString(this.getPubKey())));
            seq.close();
            return baos.toByteArray();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private ECKey(BigInteger privKey, byte[] pubKey) {
        this.priv = privKey;
        this.pub = null;
        if (pubKey == null && privKey != null) {
            this.pub = ECKey.publicKeyFromPrivate(privKey);
        } else if (pubKey != null) {
            this.pub = pubKey;
        }
    }

    public ECKey(BigInteger privKey) {
        this(privKey, (byte[])null);
    }

    public ECKey(BigInteger privKey, BigInteger pubKey) {
        this(privKey, Utils.bigIntegerToBytes(pubKey, 65));
    }

    public ECKey(byte[] privKeyBytes, byte[] pubKey) {
        this(privKeyBytes == null ? null : new BigInteger(1, privKeyBytes), pubKey);
    }

    public static byte[] publicKeyFromPrivate(BigInteger privKey) {
        return ecParams.getG().multiply(privKey).getEncoded();
    }

    public byte[] getPubKeyHash() {
        if (this.pubKeyHash == null) {
            this.pubKeyHash = Utils.sha256hash160(this.pub);
        }
        return this.pubKeyHash;
    }

    public byte[] getPubKey() {
        return this.pub;
    }

    public String toString() {
        StringBuffer b = new StringBuffer();
        b.append("pub:").append(Utils.bytesToHexString(this.pub));
        if (this.creationTimeSeconds != 0L) {
            b.append(" timestamp:" + this.creationTimeSeconds);
        }
        return b.toString();
    }

    public String toStringWithPrivate() {
        StringBuffer b = new StringBuffer();
        b.append(this.toString());
        if (this.priv != null) {
            b.append(" priv:").append(Utils.bytesToHexString(this.priv.toByteArray()));
        }
        return b.toString();
    }

    public Address toAddress(NetworkParameters params) {
        byte[] hash160 = Utils.sha256hash160(this.pub);
        return new Address(params, hash160);
    }

    public byte[] sign(byte[] input) {
        if (this.priv == null) {
            throw new IllegalStateException("This ECKey does not have the private key necessary for signing.");
        }
        ECDSASigner signer = new ECDSASigner();
        ECPrivateKeyParameters privKey = new ECPrivateKeyParameters(this.priv, ecParams);
        signer.init(true, privKey);
        BigInteger[] sigs = signer.generateSignature(input);
        try {
            UnsafeByteArrayOutputStream bos = new UnsafeByteArrayOutputStream(72);
            DERSequenceGenerator seq = new DERSequenceGenerator(bos);
            seq.addObject(new DERInteger(sigs[0]));
            seq.addObject(new DERInteger(sigs[1]));
            seq.close();
            return ((ByteArrayOutputStream)bos).toByteArray();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static boolean verify(byte[] data, byte[] signature, byte[] pub) {
        ECDSASigner signer = new ECDSASigner();
        ECPublicKeyParameters params = new ECPublicKeyParameters(ecParams.getCurve().decodePoint(pub), ecParams);
        signer.init(false, params);
        try {
            ASN1InputStream decoder = new ASN1InputStream(signature);
            DLSequence seq = (DLSequence)decoder.readObject();
            DERInteger r = (DERInteger)seq.getObjectAt(0);
            DERInteger s = (DERInteger)seq.getObjectAt(1);
            decoder.close();
            return signer.verifySignature(data, r.getPositiveValue(), s.getPositiveValue());
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public boolean verify(byte[] data, byte[] signature) {
        return ECKey.verify(data, signature, this.pub);
    }

    private static BigInteger extractPrivateKeyFromASN1(byte[] asn1privkey) {
        try {
            ASN1InputStream decoder = new ASN1InputStream(asn1privkey);
            DLSequence seq = (DLSequence)decoder.readObject();
            Preconditions.checkArgument(seq.size() == 4, "Input does not appear to be an ASN.1 OpenSSL EC private key");
            Preconditions.checkArgument(((DERInteger)seq.getObjectAt(0)).getValue().equals(BigInteger.ONE), "Input is of wrong version");
            ASN1Encodable obj = seq.getObjectAt(1);
            byte[] bits = ((ASN1OctetString)obj).getOctets();
            decoder.close();
            return new BigInteger(1, bits);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public byte[] getPrivKeyBytes() {
        return Utils.bigIntegerToBytes(this.priv, 32);
    }

    public DumpedPrivateKey getPrivateKeyEncoded(NetworkParameters params) {
        return new DumpedPrivateKey(params, this.getPrivKeyBytes());
    }

    public long getCreationTimeSeconds() {
        return this.creationTimeSeconds;
    }

    public void setCreationTimeSeconds(long newCreationTimeSeconds) {
        if (newCreationTimeSeconds < 0L) {
            throw new IllegalArgumentException("Cannot set creation time to negative value: " + newCreationTimeSeconds);
        }
        this.creationTimeSeconds = newCreationTimeSeconds;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        ECKey ecKey = (ECKey)o;
        return Arrays.equals(this.pub, ecKey.pub);
    }

    public int hashCode() {
        return this.pub[0] & 0xFF | (this.pub[1] & 0xFF) << 8 | (this.pub[2] & 0xFF) << 16 | (this.pub[3] & 0xFF) << 24;
    }

    static {
        X9ECParameters params = SECNamedCurves.getByName("secp256k1");
        ecParams = new ECDomainParameters(params.getCurve(), params.getG(), params.getN(), params.getH());
        secureRandom = new SecureRandom();
    }
}

