/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.store;

import com.google.bitcoin.core.Block;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.ProtocolException;
import com.google.bitcoin.core.Sha256Hash;
import com.google.bitcoin.core.StoredBlock;
import com.google.bitcoin.core.Utils;
import com.google.bitcoin.core.VerificationException;
import com.google.bitcoin.store.BlockStore;
import com.google.bitcoin.store.BlockStoreException;
import java.math.BigInteger;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DerbyBlockStore
implements BlockStore {
    private static final int COMMIT_INTERVAL = 2000;
    private static final Logger log = LoggerFactory.getLogger(DerbyBlockStore.class);
    private StoredBlock chainHeadBlock;
    private Sha256Hash chainHeadHash;
    private NetworkParameters params;
    private Connection conn;
    private String dbName;
    private Thread committerThread;
    static final String driver = "org.apache.derby.jdbc.EmbeddedDriver";
    static final String CREATE_SETTINGS_TABLE = "CREATE TABLE settings ( name VARCHAR(32) NOT NULL CONSTRAINT settings_pk PRIMARY KEY,value BLOB)";
    static final String CREATE_BLOCKS_TABLE = "CREATE TABLE blocks ( hash CHAR(32) FOR BIT DATA NOT NULL CONSTRAINT blocks_pk PRIMARY KEY,chainWork BLOB NOT NULL,height BIGINT NOT NULL,header BLOB NOT NULL)";
    static final String CHAIN_HEAD_SETTING = "chainhead";

    public static void main(String[] args) throws Exception {
        DerbyBlockStore store = new DerbyBlockStore(NetworkParameters.testNet(), ".bitcoinj-test.derby");
        store.resetStore();
    }

    public synchronized void close() {
        String connectionURL = "jdbc:derby:" + this.dbName + ";shutdown=true";
        try {
            if (this.conn != null) {
                this.conn.commit();
                this.conn = null;
            }
            if (this.committerThread != null) {
                this.committerThread.interrupt();
            }
            DriverManager.getConnection(connectionURL);
        }
        catch (SQLException ex) {
            if (ex.getErrorCode() == 45000 && "08006".equals(ex.getSQLState())) {
                log.info("Derby shut down normally");
            }
            throw new RuntimeException(ex);
        }
    }

    private synchronized void commit() throws BlockStoreException {
        try {
            if (this.conn != null) {
                this.conn.commit();
            }
        }
        catch (SQLException ex) {
            log.error("commit failed", ex);
            throw new BlockStoreException(ex);
        }
    }

    public DerbyBlockStore(NetworkParameters params, String dbName) throws BlockStoreException {
        this.params = params;
        this.dbName = dbName;
        String connectionURL = "jdbc:derby:" + dbName + ";create=true";
        try {
            Class.forName(driver);
            log.info("org.apache.derby.jdbc.EmbeddedDriver loaded. ");
        }
        catch (ClassNotFoundException e) {
            log.error("check CLASSPATH for Derby jar ", e);
        }
        try {
            this.conn = DriverManager.getConnection(connectionURL);
            this.conn.setAutoCommit(false);
            log.info("Connected to database " + connectionURL);
            if (!this.isTableExists("settings")) {
                this.createTables();
            }
            this.initFromDatabase();
        }
        catch (SQLException ex) {
            throw new BlockStoreException(ex);
        }
    }

    public void resetStore() throws BlockStoreException {
        try {
            Statement s = this.conn.createStatement();
            s.executeUpdate("DROP TABLE settings");
            s.executeUpdate("DROP TABLE blocks");
            s.close();
            this.createTables();
            this.initFromDatabase();
            this.startCommitter();
        }
        catch (SQLException ex) {
            throw new RuntimeException(ex);
        }
    }

    private void createTables() throws SQLException, BlockStoreException {
        Statement s = this.conn.createStatement();
        log.debug("DerbyBlockStore : CREATE blocks table");
        s.executeUpdate(CREATE_BLOCKS_TABLE);
        log.debug("DerbyBlockStore : CREATE settings table");
        s.executeUpdate(CREATE_SETTINGS_TABLE);
        s.executeUpdate("INSERT INTO settings(name, value) VALUES('chainhead', NULL)");
        this.createNewStore(this.params);
    }

    private void initFromDatabase() throws SQLException, BlockStoreException {
        Statement s = this.conn.createStatement();
        ResultSet rs = s.executeQuery("SELECT value FROM settings WHERE name = 'chainhead'");
        if (!rs.next()) {
            throw new BlockStoreException("corrupt Derby block store - no chain head pointer");
        }
        Sha256Hash hash = new Sha256Hash(rs.getBytes(1));
        this.chainHeadBlock = this.get(hash);
        if (this.chainHeadBlock == null) {
            throw new BlockStoreException("corrupt Derby block store - head block not found");
        }
        this.chainHeadHash = hash;
    }

    private void createNewStore(NetworkParameters params) throws BlockStoreException {
        try {
            StoredBlock storedGenesis;
            Block genesis = params.genesisBlock.cloneAsHeader();
            this.chainHeadBlock = storedGenesis = new StoredBlock(genesis, genesis.getWork(), 0);
            this.chainHeadHash = storedGenesis.getHeader().getHash();
            this.setChainHead(storedGenesis);
            this.put(storedGenesis);
        }
        catch (VerificationException e1) {
            throw new RuntimeException(e1);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean isTableExists(String table) throws SQLException {
        Statement s = this.conn.createStatement();
        try {
            ResultSet results = s.executeQuery("SELECT * FROM " + table + " WHERE 1 = 2");
            results.close();
            boolean bl = true;
            return bl;
        }
        catch (SQLException ex) {
            boolean bl = false;
            return bl;
        }
        finally {
            s.close();
        }
    }

    public void put(StoredBlock stored) throws BlockStoreException {
        try {
            PreparedStatement s = this.conn.prepareStatement("INSERT INTO blocks(hash, chainWork, height, header) VALUES(?, ?, ?, ?)");
            s.setBytes(1, stored.getHeader().getHash().getBytes());
            s.setBytes(2, stored.getChainWork().toByteArray());
            s.setLong(3, stored.getHeight());
            s.setBytes(4, stored.getHeader().unsafeBitcoinSerialize());
            s.executeUpdate();
            s.close();
            this.startCommitter();
        }
        catch (SQLException ex) {
            throw new BlockStoreException(ex);
        }
    }

    public StoredBlock get(Sha256Hash hash) throws BlockStoreException {
        if (this.chainHeadHash != null && this.chainHeadHash.equals(hash)) {
            return this.chainHeadBlock;
        }
        try {
            PreparedStatement s = this.conn.prepareStatement("SELECT chainWork, height, header FROM blocks WHERE hash = ?");
            s.setBytes(1, hash.getBytes());
            ResultSet results = s.executeQuery();
            if (!results.next()) {
                return null;
            }
            BigInteger chainWork = new BigInteger(results.getBytes(1));
            int height = results.getInt(2);
            Block b = new Block(this.params, results.getBytes(3));
            b.verifyHeader();
            StoredBlock stored = new StoredBlock(b, chainWork, height);
            return stored;
        }
        catch (SQLException ex) {
            throw new BlockStoreException(ex);
        }
        catch (ProtocolException e) {
            throw new BlockStoreException(e);
        }
        catch (VerificationException e) {
            throw new BlockStoreException(e);
        }
    }

    public StoredBlock getChainHead() throws BlockStoreException {
        return this.chainHeadBlock;
    }

    public void setChainHead(StoredBlock chainHead) throws BlockStoreException {
        Sha256Hash hash;
        this.chainHeadHash = hash = chainHead.getHeader().getHash();
        this.chainHeadBlock = chainHead;
        try {
            PreparedStatement s = this.conn.prepareStatement("UPDATE settings SET value = ? WHERE name = ?");
            s.setString(2, CHAIN_HEAD_SETTING);
            s.setBytes(1, hash.getBytes());
            s.executeUpdate();
            s.close();
            this.startCommitter();
        }
        catch (SQLException ex) {
            throw new BlockStoreException(ex);
        }
    }

    public void dump() throws SQLException {
        Statement s = this.conn.createStatement();
        System.out.println("settings");
        ResultSet rs = s.executeQuery("SELECT name, value FROM settings");
        while (rs.next()) {
            System.out.print(rs.getString(1));
            System.out.print(" ");
            System.out.println(Utils.bytesToHexString(rs.getBytes(2)));
        }
        rs.close();
        System.out.println("blocks");
        rs = s.executeQuery("SELECT hash, chainWork, height, header FROM blocks");
        while (rs.next()) {
            System.out.print(Utils.bytesToHexString(rs.getBytes(1)));
            System.out.print(" ");
            System.out.print(Utils.bytesToHexString(rs.getBytes(2)));
            System.out.print(" ");
            System.out.print(rs.getInt(3));
            System.out.print(" ");
            System.out.println();
        }
        rs.close();
        System.out.println("end");
        s.close();
    }

    protected synchronized void startCommitter() {
        if (this.committerThread != null) {
            return;
        }
        Runnable committer = new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            public void run() {
                try {
                    log.info("commit scheduled");
                    Thread.sleep(2000L);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                DerbyBlockStore derbyBlockStore = DerbyBlockStore.this;
                synchronized (derbyBlockStore) {
                    try {
                        if (DerbyBlockStore.this.conn != null) {
                            DerbyBlockStore.this.commit();
                            log.info("commit success");
                        } else {
                            log.info("committer noticed that we are shutting down");
                        }
                    }
                    catch (BlockStoreException e) {
                        log.warn("commit failed");
                    }
                    finally {
                        DerbyBlockStore.this.committerThread = null;
                    }
                }
            }
        };
        this.committerThread = new Thread(committer, "DerbyBlockStore committer");
        this.committerThread.start();
    }
}

