/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.store;

import com.google.bitcoin.core.ECKey;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.PeerAddress;
import com.google.bitcoin.core.Sha256Hash;
import com.google.bitcoin.core.Transaction;
import com.google.bitcoin.core.TransactionConfidence;
import com.google.bitcoin.core.TransactionInput;
import com.google.bitcoin.core.TransactionOutPoint;
import com.google.bitcoin.core.TransactionOutput;
import com.google.bitcoin.core.Wallet;
import com.google.bitcoin.core.WalletTransaction;
import com.google.bitcoin.store.WalletExtensionSerializer;
import com.google.common.base.Preconditions;
import com.google.protobuf.ByteString;
import com.google.protobuf.Message;
import com.google.protobuf.TextFormat;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import org.bitcoinj.wallet.Protos;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WalletProtobufSerializer {
    private static final Logger log = LoggerFactory.getLogger(WalletProtobufSerializer.class);
    private Map<ByteString, Transaction> txMap = new HashMap<ByteString, Transaction>();
    private WalletExtensionSerializer helper = new WalletExtensionSerializer();
    private int chainHeight;

    public void setWalletExtensionSerializer(WalletExtensionSerializer h) {
        this.helper = h;
    }

    public void writeWallet(Wallet wallet, OutputStream output) throws IOException {
        Protos.Wallet walletProto = this.walletToProto(wallet);
        walletProto.writeTo(output);
    }

    public String walletToText(Wallet wallet) {
        Protos.Wallet walletProto = this.walletToProto(wallet);
        return TextFormat.printToString((Message)walletProto);
    }

    public Protos.Wallet walletToProto(Wallet wallet) {
        Protos.Wallet.Builder walletBuilder = Protos.Wallet.newBuilder();
        walletBuilder.setNetworkIdentifier(wallet.getNetworkParameters().getId());
        for (WalletTransaction wtx : wallet.getWalletTransactions()) {
            Protos.Transaction txProto = WalletProtobufSerializer.makeTxProto(wtx);
            walletBuilder.addTransaction(txProto);
        }
        for (ECKey key : wallet.getKeys()) {
            Protos.Key.Builder buf = Protos.Key.newBuilder().setCreationTimestamp(key.getCreationTimeSeconds() * 1000L).setType(Protos.Key.Type.ORIGINAL);
            if (key.getPrivKeyBytes() != null) {
                buf.setPrivateKey(ByteString.copyFrom((byte[])key.getPrivKeyBytes()));
            }
            buf.setPublicKey(ByteString.copyFrom((byte[])key.getPubKey()));
            walletBuilder.addKey(buf);
        }
        Sha256Hash lastSeenBlockHash = wallet.getLastBlockSeenHash();
        if (lastSeenBlockHash != null) {
            walletBuilder.setLastSeenBlockHash(WalletProtobufSerializer.hashToByteString(lastSeenBlockHash));
        }
        Collection<Protos.Extension> extensions = this.helper.getExtensionsToWrite(wallet);
        for (Protos.Extension ext : extensions) {
            walletBuilder.addExtension(ext);
        }
        return walletBuilder.build();
    }

    private static Protos.Transaction makeTxProto(WalletTransaction wtx) {
        Transaction tx = wtx.getTransaction();
        Protos.Transaction.Builder txBuilder = Protos.Transaction.newBuilder();
        txBuilder.setPool(Protos.Transaction.Pool.valueOf(wtx.getPool().getValue())).setHash(WalletProtobufSerializer.hashToByteString(tx.getHash())).setVersion((int)tx.getVersion());
        if (tx.getUpdateTime() != null) {
            txBuilder.setUpdatedAt(tx.getUpdateTime().getTime());
        }
        if (tx.getLockTime() > 0L) {
            txBuilder.setLockTime((int)tx.getLockTime());
        }
        for (TransactionInput input : tx.getInputs()) {
            Protos.TransactionInput.Builder inputBuilder = Protos.TransactionInput.newBuilder().setScriptBytes(ByteString.copyFrom((byte[])input.getScriptBytes())).setTransactionOutPointHash(WalletProtobufSerializer.hashToByteString(input.getOutpoint().getHash())).setTransactionOutPointIndex((int)input.getOutpoint().getIndex());
            if (input.hasSequence()) {
                inputBuilder.setSequence((int)input.getSequence());
            }
            txBuilder.addTransactionInput(inputBuilder);
        }
        for (TransactionOutput output : tx.getOutputs()) {
            Protos.TransactionOutput.Builder outputBuilder = Protos.TransactionOutput.newBuilder().setScriptBytes(ByteString.copyFrom((byte[])output.getScriptBytes())).setValue(output.getValue().longValue());
            TransactionInput spentBy = output.getSpentBy();
            if (spentBy != null) {
                Sha256Hash spendingHash = spentBy.getParentTransaction().getHash();
                int spentByTransactionIndex = spentBy.getParentTransaction().getInputs().indexOf(spentBy);
                outputBuilder.setSpentByTransactionHash(WalletProtobufSerializer.hashToByteString(spendingHash)).setSpentByTransactionIndex(spentByTransactionIndex);
            }
            txBuilder.addTransactionOutput(outputBuilder);
        }
        if (tx.getAppearsInHashes() != null) {
            for (Sha256Hash hash : tx.getAppearsInHashes()) {
                txBuilder.addBlockHash(WalletProtobufSerializer.hashToByteString(hash));
            }
        }
        if (tx.hasConfidence()) {
            TransactionConfidence confidence = tx.getConfidence();
            Protos.TransactionConfidence.Builder confidenceBuilder = Protos.TransactionConfidence.newBuilder();
            WalletProtobufSerializer.writeConfidence(txBuilder, confidence, confidenceBuilder);
        }
        return txBuilder.build();
    }

    private static void writeConfidence(Protos.Transaction.Builder txBuilder, TransactionConfidence confidence, Protos.TransactionConfidence.Builder confidenceBuilder) {
        confidenceBuilder.setType(Protos.TransactionConfidence.Type.valueOf(confidence.getConfidenceType().getValue()));
        if (confidence.getConfidenceType() == TransactionConfidence.ConfidenceType.BUILDING) {
            confidenceBuilder.setAppearedAtHeight(confidence.getAppearedAtChainHeight());
            confidenceBuilder.setDepth(confidence.getDepthInBlocks());
            if (confidence.getWorkDone() != null) {
                confidenceBuilder.setWorkDone(confidence.getWorkDone().longValue());
            }
        }
        if (confidence.getConfidenceType() == TransactionConfidence.ConfidenceType.DEAD) {
            Sha256Hash overridingHash = confidence.getOverridingTransaction().getHash();
            confidenceBuilder.setOverridingTransaction(WalletProtobufSerializer.hashToByteString(overridingHash));
        }
        for (PeerAddress address : confidence.getBroadcastBy()) {
            Protos.PeerAddress proto = Protos.PeerAddress.newBuilder().setIpAddress(ByteString.copyFrom((byte[])address.getAddr().getAddress())).setPort(address.getPort()).setServices(address.getServices().longValue()).build();
            confidenceBuilder.addBroadcastBy(proto);
        }
        txBuilder.setConfidence(confidenceBuilder);
    }

    private static ByteString hashToByteString(Sha256Hash hash) {
        return ByteString.copyFrom((byte[])hash.getBytes());
    }

    private static Sha256Hash byteStringToHash(ByteString bs) {
        return new Sha256Hash(bs.toByteArray());
    }

    public void setChainHeight(int chainHeight) {
        this.chainHeight = chainHeight;
    }

    public Wallet readWallet(InputStream input) throws IOException {
        Protos.Wallet walletProto = WalletProtobufSerializer.parseToProto(input);
        NetworkParameters params = NetworkParameters.fromID(walletProto.getNetworkIdentifier());
        Wallet wallet = this.helper.newWallet(params);
        for (Protos.Key keyProto : walletProto.getKeyList()) {
            if (keyProto.getType() != Protos.Key.Type.ORIGINAL) {
                throw new IllegalArgumentException("Unknown key type in wallet");
            }
            byte[] privKey = null;
            if (keyProto.hasPrivateKey()) {
                privKey = keyProto.getPrivateKey().toByteArray();
            }
            byte[] pubKey = keyProto.hasPublicKey() ? keyProto.getPublicKey().toByteArray() : null;
            ECKey ecKey = new ECKey(privKey, pubKey);
            ecKey.setCreationTimeSeconds((keyProto.getCreationTimestamp() + 500L) / 1000L);
            wallet.addKey(ecKey);
        }
        for (Protos.Transaction txProto : walletProto.getTransactionList()) {
            this.readTransaction(txProto, params);
        }
        for (Protos.Transaction txProto : walletProto.getTransactionList()) {
            WalletTransaction wtx = this.connectTransactionOutputs(txProto);
            wallet.addWalletTransaction(wtx);
        }
        if (!walletProto.hasLastSeenBlockHash()) {
            wallet.setLastBlockSeenHash(null);
        } else {
            wallet.setLastBlockSeenHash(WalletProtobufSerializer.byteStringToHash(walletProto.getLastSeenBlockHash()));
        }
        for (Protos.Extension extProto : walletProto.getExtensionList()) {
            this.helper.readExtension(wallet, extProto);
        }
        return wallet;
    }

    public static Protos.Wallet parseToProto(InputStream input) throws IOException {
        return Protos.Wallet.parseFrom(input);
    }

    private void readTransaction(Protos.Transaction txProto, NetworkParameters params) {
        Transaction tx = new Transaction(params);
        if (txProto.hasUpdatedAt()) {
            tx.setUpdateTime(new Date(txProto.getUpdatedAt()));
        }
        for (Protos.TransactionOutput outputProto : txProto.getTransactionOutputList()) {
            BigInteger value = BigInteger.valueOf(outputProto.getValue());
            byte[] scriptBytes = outputProto.getScriptBytes().toByteArray();
            TransactionOutput output = new TransactionOutput(params, tx, value, scriptBytes);
            tx.addOutput(output);
        }
        for (Protos.TransactionInput transactionInput : txProto.getTransactionInputList()) {
            byte[] scriptBytes = transactionInput.getScriptBytes().toByteArray();
            TransactionOutPoint outpoint = new TransactionOutPoint(params, (long)transactionInput.getTransactionOutPointIndex(), WalletProtobufSerializer.byteStringToHash(transactionInput.getTransactionOutPointHash()));
            TransactionInput input = new TransactionInput(params, tx, scriptBytes, outpoint);
            if (transactionInput.hasSequence()) {
                input.setSequence(transactionInput.getSequence());
            }
            tx.addInput(input);
        }
        for (ByteString blockHash : txProto.getBlockHashList()) {
            tx.addBlockAppearance(WalletProtobufSerializer.byteStringToHash(blockHash));
        }
        if (txProto.hasLockTime()) {
            tx.setLockTime(txProto.getLockTime());
        }
        Sha256Hash protoHash = WalletProtobufSerializer.byteStringToHash(txProto.getHash());
        Preconditions.checkState(tx.getHash().equals(protoHash), "Transaction did not deserialize completely: %s vs %s", tx.getHash(), protoHash);
        Preconditions.checkState(!this.txMap.containsKey(txProto.getHash()), "Wallet contained duplicate transaction %s", WalletProtobufSerializer.byteStringToHash(txProto.getHash()));
        this.txMap.put(txProto.getHash(), tx);
    }

    private WalletTransaction connectTransactionOutputs(Protos.Transaction txProto) {
        Transaction tx = this.txMap.get(txProto.getHash());
        WalletTransaction.Pool pool = WalletTransaction.Pool.valueOf(txProto.getPool().getNumber());
        for (int i = 0; i < tx.getOutputs().size(); ++i) {
            TransactionOutput output = tx.getOutputs().get(i);
            Protos.TransactionOutput transactionOutput = txProto.getTransactionOutput(i);
            if (!transactionOutput.hasSpentByTransactionHash()) continue;
            Transaction spendingTx = this.txMap.get(transactionOutput.getSpentByTransactionHash());
            int spendingIndex = transactionOutput.getSpentByTransactionIndex();
            TransactionInput input = spendingTx.getInputs().get(spendingIndex);
            input.connect(output);
        }
        if (txProto.hasConfidence()) {
            Protos.TransactionConfidence confidenceProto = txProto.getConfidence();
            TransactionConfidence confidence = tx.getConfidence();
            this.readConfidence(tx, confidenceProto, confidence);
        }
        return new WalletTransaction(pool, tx);
    }

    private void readConfidence(Transaction tx, Protos.TransactionConfidence confidenceProto, TransactionConfidence confidence) {
        if (!confidenceProto.hasType()) {
            log.warn("Unknown confidence type for tx {}", (Object)tx.getHashAsString());
            return;
        }
        TransactionConfidence.ConfidenceType confidenceType = TransactionConfidence.ConfidenceType.valueOf(confidenceProto.getType().getNumber());
        confidence.setConfidenceType(confidenceType);
        if (confidenceProto.hasAppearedAtHeight()) {
            if (confidence.getConfidenceType() != TransactionConfidence.ConfidenceType.BUILDING) {
                log.warn("Have appearedAtHeight but not BUILDING for tx {}", (Object)tx.getHashAsString());
                return;
            }
            confidence.setAppearedAtChainHeight(confidenceProto.getAppearedAtHeight());
        }
        if (confidenceProto.hasDepth()) {
            if (confidence.getConfidenceType() != TransactionConfidence.ConfidenceType.BUILDING) {
                log.warn("Have depth but not BUILDING for tx {}", (Object)tx.getHashAsString());
                return;
            }
            confidence.setDepthInBlocks(confidenceProto.getDepth());
        } else if (this.chainHeight != 0 && confidenceProto.hasAppearedAtHeight()) {
            confidence.setDepthInBlocks(this.chainHeight - confidence.getAppearedAtChainHeight() + 1);
        }
        if (confidenceProto.hasWorkDone()) {
            if (confidence.getConfidenceType() != TransactionConfidence.ConfidenceType.BUILDING) {
                log.warn("Have workDone but not BUILDING for tx {}", (Object)tx.getHashAsString());
                return;
            }
            confidence.setWorkDone(BigInteger.valueOf(confidenceProto.getWorkDone()));
        }
        if (confidenceProto.hasOverridingTransaction()) {
            if (confidence.getConfidenceType() != TransactionConfidence.ConfidenceType.DEAD) {
                log.warn("Have overridingTransaction but not OVERRIDDEN for tx {}", (Object)tx.getHashAsString());
                return;
            }
            Transaction overridingTransaction = this.txMap.get(confidenceProto.getOverridingTransaction());
            if (overridingTransaction == null) {
                log.warn("Have overridingTransaction that is not in wallet for tx {}", (Object)tx.getHashAsString());
                return;
            }
            confidence.setOverridingTransaction(overridingTransaction);
        }
        for (Protos.PeerAddress proto : confidenceProto.getBroadcastByList()) {
            InetAddress ip;
            try {
                ip = InetAddress.getByAddress(proto.getIpAddress().toByteArray());
            }
            catch (UnknownHostException e) {
                throw new RuntimeException(e);
            }
            int port = proto.getPort();
            PeerAddress address = new PeerAddress(ip, port);
            address.setServices(BigInteger.valueOf(proto.getServices()));
            confidence.markBroadcastBy(address);
        }
    }
}

