/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.core;

import com.google.bitcoin.core.AbstractPeerEventListener;
import com.google.bitcoin.core.Address;
import com.google.bitcoin.core.Block;
import com.google.bitcoin.core.BlockChain;
import com.google.bitcoin.core.ECKey;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.Peer;
import com.google.bitcoin.core.PeerEventListener;
import com.google.bitcoin.core.PeerGroup;
import com.google.bitcoin.core.ScriptException;
import com.google.bitcoin.core.Sha256Hash;
import com.google.bitcoin.core.StoredBlock;
import com.google.bitcoin.core.Transaction;
import com.google.bitcoin.core.TransactionConfidence;
import com.google.bitcoin.core.TransactionInput;
import com.google.bitcoin.core.TransactionOutPoint;
import com.google.bitcoin.core.TransactionOutput;
import com.google.bitcoin.core.Utils;
import com.google.bitcoin.core.VerificationException;
import com.google.bitcoin.core.WalletEventListener;
import com.google.bitcoin.core.WalletTransaction;
import com.google.bitcoin.store.WalletProtobufSerializer;
import com.google.bitcoin.utils.EventListenerInvoker;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ListenableFuture;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class Wallet
implements Serializable {
    private static final Logger log = LoggerFactory.getLogger(Wallet.class);
    private static final long serialVersionUID = 2L;
    final Map<Sha256Hash, Transaction> pending;
    final Map<Sha256Hash, Transaction> unspent;
    final Map<Sha256Hash, Transaction> spent;
    final Map<Sha256Hash, Transaction> inactive;
    final Map<Sha256Hash, Transaction> dead;
    public final ArrayList<ECKey> keychain;
    private final NetworkParameters params;
    private Sha256Hash lastBlockSeenHash;
    private transient ArrayList<WalletEventListener> eventListeners;
    private transient File autosaveToFile;
    private transient boolean dirty;
    private transient AutosaveEventListener autosaveEventListener;
    private transient long autosaveDelayMs;
    private transient TransactionConfidence.Listener txConfidenceListener;
    private transient HashSet<Sha256Hash> ignoreNextNewBlock;
    private transient PeerEventListener peerEventListener;

    public Wallet(NetworkParameters params) {
        this.params = params;
        this.keychain = new ArrayList();
        this.unspent = new HashMap<Sha256Hash, Transaction>();
        this.spent = new HashMap<Sha256Hash, Transaction>();
        this.inactive = new HashMap<Sha256Hash, Transaction>();
        this.pending = new HashMap<Sha256Hash, Transaction>();
        this.dead = new HashMap<Sha256Hash, Transaction>();
        this.createTransientState();
    }

    private void createTransientState() {
        this.eventListeners = new ArrayList();
        this.ignoreNextNewBlock = new HashSet();
        this.txConfidenceListener = new TransactionConfidence.Listener(){

            public void onConfidenceChanged(Transaction tx) {
                Wallet.this.invokeOnTransactionConfidenceChanged(tx);
            }
        };
    }

    public NetworkParameters getNetworkParameters() {
        return this.params;
    }

    public synchronized Iterable<ECKey> getKeys() {
        return new ArrayList<ECKey>(this.keychain);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void saveToFile(File temp, File destFile) throws IOException {
        FileOutputStream stream = null;
        try {
            stream = new FileOutputStream(temp);
            this.saveToFileStream(stream);
            stream.flush();
            stream.getFD().sync();
            stream.close();
            stream = null;
            if (!temp.renameTo(destFile)) {
                if (System.getProperty("os.name").toLowerCase().indexOf("win") >= 0 && destFile.delete() && temp.renameTo(destFile)) {
                    return;
                }
                throw new IOException("Failed to rename " + temp + " to " + destFile);
            }
            if (destFile.equals(this.autosaveToFile)) {
                this.dirty = false;
            }
        }
        finally {
            if (stream != null) {
                stream.close();
            }
        }
    }

    public synchronized void saveToFile(File f) throws IOException {
        File directory = f.getAbsoluteFile().getParentFile();
        File temp = File.createTempFile("wallet", null, directory);
        this.saveToFile(temp, f);
    }

    private synchronized boolean autoSave() {
        try {
            log.info("Auto-saving wallet, last seen block is {}", this.lastBlockSeenHash);
            File directory = this.autosaveToFile.getAbsoluteFile().getParentFile();
            File temp = File.createTempFile("wallet", null, directory);
            if (this.autosaveEventListener != null) {
                this.autosaveEventListener.onBeforeAutoSave(temp);
            }
            this.saveToFile(temp, this.autosaveToFile);
            if (this.autosaveEventListener != null) {
                this.autosaveEventListener.onAfterAutoSave(this.autosaveToFile);
            }
        }
        catch (Exception e) {
            if (this.autosaveEventListener != null && this.autosaveEventListener.caughtException(e)) {
                return true;
            }
            throw new RuntimeException(e);
        }
        return false;
    }

    public synchronized void autosaveToFile(File f, long delayTime, TimeUnit timeUnit, AutosaveEventListener eventListener) {
        Preconditions.checkArgument(delayTime >= 0L);
        this.autosaveToFile = Preconditions.checkNotNull(f);
        if (delayTime > 0L) {
            this.autosaveEventListener = eventListener;
            this.autosaveDelayMs = TimeUnit.MILLISECONDS.convert(delayTime, timeUnit);
        }
    }

    private synchronized void queueAutoSave() {
        if (this.autosaveToFile == null) {
            return;
        }
        if (this.autosaveDelayMs == 0L) {
            try {
                this.saveToFile(this.autosaveToFile);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        } else if (!this.dirty) {
            this.dirty = true;
            AutosaveThread.registerForSave(this, this.autosaveDelayMs);
        }
    }

    public synchronized void saveToFileStream(OutputStream f) throws IOException {
        new WalletProtobufSerializer().writeWallet(this, f);
    }

    public NetworkParameters getParams() {
        return this.params;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static Wallet loadFromFile(File f) throws IOException {
        FileInputStream stream = new FileInputStream(f);
        try {
            Wallet wallet = Wallet.loadFromFileStream(stream);
            return wallet;
        }
        finally {
            stream.close();
        }
    }

    public boolean isConsistent() {
        int size2;
        boolean success = true;
        HashSet<Transaction> pendingInactive = new HashSet<Transaction>();
        pendingInactive.addAll(this.pending.values());
        pendingInactive.addAll(this.inactive.values());
        Set<Transaction> transactions = this.getTransactions(true, true);
        HashSet<Sha256Hash> hashes = new HashSet<Sha256Hash>();
        for (Transaction tx : transactions) {
            hashes.add(tx.getHash());
        }
        int size1 = transactions.size();
        if (size1 != hashes.size()) {
            log.error("Two transactions with same hash");
            success = false;
        }
        if (size1 != (size2 = this.unspent.size() + this.spent.size() + pendingInactive.size() + this.dead.size())) {
            log.error("Inconsistent wallet sizes: {} {}", size1, (Object)size2);
            success = false;
        }
        for (Transaction tx : this.unspent.values()) {
            if (tx.isConsistent(this, false)) continue;
            success = false;
            log.error("Inconsistent unspent tx {}", (Object)tx.getHashAsString());
        }
        for (Transaction tx : this.spent.values()) {
            if (tx.isConsistent(this, true)) continue;
            success = false;
            log.error("Inconsistent spent tx {}", (Object)tx.getHashAsString());
        }
        return success;
    }

    public static Wallet loadFromFileStream(InputStream stream) throws IOException {
        Wallet wallet;
        stream = new BufferedInputStream(stream);
        stream.mark(100);
        boolean serialization = stream.read() == 172 && stream.read() == 237;
        stream.reset();
        if (serialization) {
            ObjectInputStream ois = null;
            try {
                ois = new ObjectInputStream(stream);
                wallet = (Wallet)ois.readObject();
            }
            catch (ClassNotFoundException e) {
                throw new RuntimeException(e);
            }
            finally {
                if (ois != null) {
                    ois.close();
                }
            }
        } else {
            wallet = new WalletProtobufSerializer().readWallet(stream);
        }
        if (!wallet.isConsistent()) {
            log.error("Loaded an inconsistent wallet");
        }
        return wallet;
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        this.createTransientState();
    }

    public synchronized void receiveFromBlock(Transaction tx, StoredBlock block, BlockChain.NewBlockType blockType) throws VerificationException, ScriptException {
        this.receive(tx, block, blockType, false);
    }

    public synchronized void receivePending(Transaction tx) throws VerificationException, ScriptException {
        TransactionConfidence.ConfidenceType currentConfidence;
        EnumSet<WalletTransaction.Pool> containingPools = this.getContainingPools(tx);
        if (!containingPools.equals(EnumSet.noneOf(WalletTransaction.Pool.class))) {
            log.debug("Received tx we already saw in a block or created ourselves: " + tx.getHashAsString());
            return;
        }
        if (!this.isTransactionRelevant(tx, true)) {
            log.debug("Received tx that isn't relevant to this wallet, discarding.");
            return;
        }
        BigInteger valueSentToMe = tx.getValueSentToMe(this);
        BigInteger valueSentFromMe = tx.getValueSentFromMe(this);
        if (log.isInfoEnabled()) {
            log.info(String.format("Received a pending transaction %s that spends %s BTC from our own wallet, and sends us %s BTC", tx.getHashAsString(), Utils.bitcoinValueToFriendlyString(valueSentFromMe), Utils.bitcoinValueToFriendlyString(valueSentToMe)));
        }
        if ((currentConfidence = tx.getConfidence().getConfidenceType()) == TransactionConfidence.ConfidenceType.UNKNOWN) {
            tx.getConfidence().setConfidenceType(TransactionConfidence.ConfidenceType.NOT_SEEN_IN_CHAIN);
            this.invokeOnTransactionConfidenceChanged(tx);
        }
        this.commitTx(tx);
    }

    private void invokeOnCoinsReceived(final Transaction tx, final BigInteger balance, final BigInteger newBalance) {
        EventListenerInvoker.invoke(this.eventListeners, new EventListenerInvoker<WalletEventListener>(){

            @Override
            public void invoke(WalletEventListener listener) {
                listener.onCoinsReceived(Wallet.this, tx, balance, newBalance);
            }
        });
    }

    private void invokeOnCoinsSent(final Transaction tx, final BigInteger prevBalance, final BigInteger newBalance) {
        EventListenerInvoker.invoke(this.eventListeners, new EventListenerInvoker<WalletEventListener>(){

            @Override
            public void invoke(WalletEventListener listener) {
                listener.onCoinsSent(Wallet.this, tx, prevBalance, newBalance);
            }
        });
    }

    public synchronized boolean isTransactionRelevant(Transaction tx, boolean includeDoubleSpending) throws ScriptException {
        return tx.getValueSentFromMe(this).compareTo(BigInteger.ZERO) > 0 || tx.getValueSentToMe(this).compareTo(BigInteger.ZERO) > 0 || includeDoubleSpending && this.findDoubleSpendAgainstPending(tx) != null;
    }

    private Transaction findDoubleSpendAgainstPending(Transaction tx) {
        HashSet<TransactionOutPoint> outpoints = new HashSet<TransactionOutPoint>();
        for (TransactionInput input : tx.getInputs()) {
            outpoints.add(input.getOutpoint());
        }
        for (Transaction p : this.pending.values()) {
            for (TransactionInput input : p.getInputs()) {
                if (!outpoints.contains(input.getOutpoint())) continue;
                return p;
            }
        }
        return null;
    }

    private synchronized void receive(Transaction tx, StoredBlock block, BlockChain.NewBlockType blockType, boolean reorg) throws VerificationException, ScriptException {
        boolean wasPending;
        Transaction wtx;
        BigInteger prevBalance = this.getBalance();
        Sha256Hash txHash = tx.getHash();
        boolean bestChain = blockType == BlockChain.NewBlockType.BEST_CHAIN;
        boolean sideChain = blockType == BlockChain.NewBlockType.SIDE_CHAIN;
        BigInteger valueSentFromMe = tx.getValueSentFromMe(this);
        BigInteger valueSentToMe = tx.getValueSentToMe(this);
        BigInteger valueDifference = valueSentToMe.subtract(valueSentFromMe);
        if (!reorg) {
            log.info("Received tx {} for {} BTC: {}", new Object[]{sideChain ? "on a side chain" : "", Utils.bitcoinValueToFriendlyString(valueDifference), tx.getHashAsString()});
        }
        if ((wtx = this.pending.remove(txHash)) != null) {
            tx = wtx;
            log.info("  <-pending");
            if (bestChain) {
                if (valueSentToMe.equals(BigInteger.ZERO)) {
                    log.info("  ->spent");
                    this.addWalletTransaction(WalletTransaction.Pool.SPENT, tx);
                } else {
                    log.info("  ->unspent");
                    this.addWalletTransaction(WalletTransaction.Pool.UNSPENT, tx);
                }
            } else if (sideChain) {
                boolean alreadyPresent;
                log.info("  ->inactive");
                boolean bl = alreadyPresent = this.inactive.put(tx.getHash(), tx) != null;
                if (alreadyPresent) {
                    log.info("Saw a transaction be incorporated into multiple independent side chains");
                }
                this.pending.put(tx.getHash(), tx);
            }
        } else if (sideChain) {
            if (!this.unspent.containsKey(tx.getHash()) && !this.spent.containsKey(tx.getHash())) {
                log.info("  ->inactive");
                this.addWalletTransaction(WalletTransaction.Pool.INACTIVE, tx);
            }
        } else if (bestChain) {
            this.processTxFromBestChain(tx);
        }
        log.info("Balance is now: " + Utils.bitcoinValueToFriendlyString(this.getBalance()));
        if (block != null) {
            tx.setBlockAppearance(block, bestChain);
            if (bestChain) {
                this.ignoreNextNewBlock.add(txHash);
            }
        }
        boolean bl = wasPending = wtx != null;
        if (!reorg && bestChain && !wasPending) {
            BigInteger newBalance = this.getBalance();
            int diff = valueDifference.compareTo(BigInteger.ZERO);
            if (diff > 0) {
                this.invokeOnCoinsReceived(tx, prevBalance, newBalance);
            } else if (diff == 0) {
                this.invokeOnCoinsSent(tx, prevBalance, newBalance);
            } else {
                this.invokeOnCoinsSent(tx, prevBalance, newBalance);
            }
        }
        Preconditions.checkState(this.isConsistent());
        this.queueAutoSave();
    }

    public synchronized void notifyNewBestBlock(Block block) throws VerificationException {
        Sha256Hash newBlockHash = block.getHash();
        if (!newBlockHash.equals(this.getLastBlockSeenHash())) {
            this.setLastBlockSeenHash(newBlockHash);
            Set<Transaction> transactions = this.getTransactions(true, false);
            for (Transaction tx : transactions) {
                if (this.ignoreNextNewBlock.contains(tx.getHash())) {
                    this.ignoreNextNewBlock.remove(tx.getHash());
                    continue;
                }
                tx.getConfidence().notifyWorkDone(block);
            }
            this.queueAutoSave();
        }
    }

    private void processTxFromBestChain(Transaction tx) throws VerificationException, ScriptException {
        boolean isDeadCoinbase;
        boolean bl = isDeadCoinbase = tx.isCoinBase() && this.dead.containsKey(tx.getHash());
        if (isDeadCoinbase) {
            log.info("  coinbase tx {} <-dead: confidence {}", (Object)tx.getHashAsString(), (Object)tx.getConfidence().getConfidenceType().name());
            this.dead.remove(tx.getHash());
        }
        if (this.inactive.containsKey(tx.getHash())) {
            log.info("  new tx {} <-inactive", (Object)tx.getHashAsString());
            this.inactive.remove(tx.getHash());
        }
        this.updateForSpends(tx, true);
        if (!tx.getValueSentToMe(this).equals(BigInteger.ZERO)) {
            log.info("  new tx {} ->unspent", (Object)tx.getHashAsString());
            this.addWalletTransaction(WalletTransaction.Pool.UNSPENT, tx);
        } else if (!tx.getValueSentFromMe(this).equals(BigInteger.ZERO)) {
            log.info("  new tx {} ->spent", (Object)tx.getHashAsString());
            this.addWalletTransaction(WalletTransaction.Pool.SPENT, tx);
        } else {
            Transaction doubleSpend = this.findDoubleSpendAgainstPending(tx);
            if (doubleSpend == null) {
                throw new IllegalStateException("Received an irrelevant tx that was not a double spend.");
            }
            log.warn("Saw double spend from chain override pending tx {}", (Object)doubleSpend.getHashAsString());
            log.warn("  <-pending ->dead");
            this.pending.remove(doubleSpend.getHash());
            this.addWalletTransaction(WalletTransaction.Pool.DEAD, doubleSpend);
            doubleSpend.getConfidence().setOverridingTransaction(tx);
        }
    }

    private void updateForSpends(Transaction tx, boolean fromChain) throws VerificationException {
        List<TransactionInput> inputs = tx.getInputs();
        for (int i = 0; i < inputs.size(); ++i) {
            TransactionInput input = inputs.get(i);
            TransactionInput.ConnectionResult result = input.connect(this.unspent, TransactionInput.ConnectMode.ABORT_ON_CONFLICT);
            if (result == TransactionInput.ConnectionResult.NO_SUCH_TX && (result = input.connect(this.spent, TransactionInput.ConnectMode.ABORT_ON_CONFLICT)) == TransactionInput.ConnectionResult.NO_SUCH_TX && (result = input.connect(this.pending, TransactionInput.ConnectMode.ABORT_ON_CONFLICT)) == TransactionInput.ConnectionResult.NO_SUCH_TX) continue;
            if (result == TransactionInput.ConnectionResult.ALREADY_SPENT) {
                Transaction doubleSpent = input.getOutpoint().fromTx;
                Preconditions.checkNotNull(doubleSpent);
                int index = (int)input.getOutpoint().getIndex();
                TransactionOutput output = doubleSpent.getOutputs().get(index);
                TransactionInput spentBy = Preconditions.checkNotNull(output.getSpentBy());
                Transaction connected = Preconditions.checkNotNull(spentBy.getParentTransaction());
                if (fromChain) {
                    if (!this.pending.containsKey(connected.getHash())) continue;
                    log.warn("Saw double spend from chain override pending tx {}", (Object)connected.getHashAsString());
                    log.warn("  <-pending ->dead");
                    this.pending.remove(connected.getHash());
                    this.dead.put(connected.getHash(), connected);
                    input.connect(this.unspent, TransactionInput.ConnectMode.DISCONNECT_ON_CONFLICT);
                    connected.getConfidence().setOverridingTransaction(tx);
                    continue;
                }
                log.warn("Saw double spend from another pending transaction, ignoring tx {}", (Object)tx.getHashAsString());
                log.warn("  offending input is input {}", i);
                return;
            }
            if (result != TransactionInput.ConnectionResult.SUCCESS) continue;
            Transaction connected = Preconditions.checkNotNull(input.getOutpoint().fromTx);
            this.maybeMoveTxToSpent(connected, "prevtx");
        }
    }

    private void maybeMoveTxToSpent(Transaction tx, String context) {
        if (tx.isEveryOwnedOutputSpent(this) && this.unspent.remove(tx.getHash()) != null) {
            if (log.isInfoEnabled()) {
                log.info("  {} {} <-unspent", (Object)tx.getHashAsString(), (Object)context);
                log.info("  {} {} ->spent", (Object)tx.getHashAsString(), (Object)context);
            }
            this.spent.put(tx.getHash(), tx);
        }
    }

    public synchronized void addEventListener(WalletEventListener listener) {
        this.eventListeners.add(listener);
    }

    public synchronized boolean removeEventListener(WalletEventListener listener) {
        return this.eventListeners.remove(listener);
    }

    public synchronized void commitTx(Transaction tx) throws VerificationException {
        Preconditions.checkArgument(!this.pending.containsKey(tx.getHash()), "commitTx called on the same transaction twice");
        log.info("commitTx of {}", (Object)tx.getHashAsString());
        BigInteger balance = this.getBalance();
        tx.updatedAt = Utils.now();
        this.updateForSpends(tx, false);
        log.info("->pending: {}", (Object)tx.getHashAsString());
        this.addWalletTransaction(WalletTransaction.Pool.PENDING, tx);
        try {
            BigInteger valueSentFromMe = tx.getValueSentFromMe(this);
            BigInteger valueSentToMe = tx.getValueSentToMe(this);
            BigInteger newBalance = balance.add(valueSentToMe).subtract(valueSentFromMe);
            if (valueSentToMe.compareTo(BigInteger.ZERO) > 0) {
                this.invokeOnCoinsReceived(tx, balance, newBalance);
            }
            if (valueSentFromMe.compareTo(BigInteger.ZERO) > 0) {
                this.invokeOnCoinsSent(tx, balance, newBalance);
            }
        }
        catch (ScriptException e) {
            throw new RuntimeException(e);
        }
        Preconditions.checkState(this.isConsistent());
        this.queueAutoSave();
    }

    public synchronized Set<Transaction> getTransactions(boolean includeDead, boolean includeInactive) {
        HashSet<Transaction> all = new HashSet<Transaction>();
        all.addAll(this.unspent.values());
        all.addAll(this.spent.values());
        all.addAll(this.pending.values());
        if (includeDead) {
            all.addAll(this.dead.values());
        }
        if (includeInactive) {
            all.addAll(this.inactive.values());
        }
        return all;
    }

    public synchronized Iterable<WalletTransaction> getWalletTransactions() {
        HashSet<Transaction> pendingInactive = new HashSet<Transaction>();
        pendingInactive.addAll(this.pending.values());
        pendingInactive.retainAll(this.inactive.values());
        HashSet<Transaction> onlyPending = new HashSet<Transaction>();
        HashSet<Transaction> onlyInactive = new HashSet<Transaction>();
        onlyPending.addAll(this.pending.values());
        onlyPending.removeAll(pendingInactive);
        onlyInactive.addAll(this.inactive.values());
        onlyInactive.removeAll(pendingInactive);
        HashSet<WalletTransaction> all = new HashSet<WalletTransaction>();
        Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.UNSPENT, this.unspent.values());
        Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.SPENT, this.spent.values());
        Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.DEAD, this.dead.values());
        Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.PENDING, onlyPending);
        Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.INACTIVE, onlyInactive);
        Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.PENDING_INACTIVE, pendingInactive);
        return all;
    }

    private static synchronized void addWalletTransactionsToSet(Set<WalletTransaction> txs, WalletTransaction.Pool poolType, Collection<Transaction> pool) {
        for (Transaction tx : pool) {
            txs.add(new WalletTransaction(poolType, tx));
        }
    }

    public void addWalletTransaction(WalletTransaction wtx) {
        this.addWalletTransaction(wtx.getPool(), wtx.getTransaction());
    }

    private synchronized void addWalletTransaction(WalletTransaction.Pool pool, Transaction tx) {
        switch (pool) {
            case UNSPENT: {
                Preconditions.checkState(this.unspent.put(tx.getHash(), tx) == null);
                break;
            }
            case SPENT: {
                Preconditions.checkState(this.spent.put(tx.getHash(), tx) == null);
                break;
            }
            case PENDING: {
                Preconditions.checkState(this.pending.put(tx.getHash(), tx) == null);
                break;
            }
            case DEAD: {
                Preconditions.checkState(this.dead.put(tx.getHash(), tx) == null);
                break;
            }
            case INACTIVE: {
                Preconditions.checkState(this.inactive.put(tx.getHash(), tx) == null);
                break;
            }
            case PENDING_INACTIVE: {
                Preconditions.checkState(this.pending.put(tx.getHash(), tx) == null);
                Preconditions.checkState(this.inactive.put(tx.getHash(), tx) == null);
                break;
            }
            default: {
                throw new RuntimeException("Unknown wallet transaction type " + (Object)((Object)pool));
            }
        }
        tx.getConfidence().addEventListener(this.txConfidenceListener);
    }

    public List<Transaction> getTransactionsByTime() {
        return this.getRecentTransactions(0, false);
    }

    public synchronized List<Transaction> getRecentTransactions(int numTransactions, boolean includeDead) {
        Preconditions.checkArgument(numTransactions >= 0);
        int size = this.getPoolSize(WalletTransaction.Pool.UNSPENT) + this.getPoolSize(WalletTransaction.Pool.SPENT) + this.getPoolSize(WalletTransaction.Pool.PENDING);
        if (numTransactions > size || numTransactions == 0) {
            numTransactions = size;
        }
        ArrayList<Transaction> all = new ArrayList<Transaction>(this.getTransactions(includeDead, false));
        Collections.sort(all, Collections.reverseOrder(new Comparator<Transaction>(){

            @Override
            public int compare(Transaction t1, Transaction t2) {
                return t1.getUpdateTime().compareTo(t2.getUpdateTime());
            }
        }));
        if (numTransactions == all.size()) {
            return all;
        }
        all.subList(numTransactions, all.size()).clear();
        return all;
    }

    public synchronized Transaction getTransaction(Sha256Hash hash) {
        Transaction tx = this.pending.get(hash);
        if (tx != null) {
            return tx;
        }
        tx = this.unspent.get(hash);
        if (tx != null) {
            return tx;
        }
        tx = this.spent.get(hash);
        if (tx != null) {
            return tx;
        }
        tx = this.inactive.get(hash);
        if (tx != null) {
            return tx;
        }
        tx = this.dead.get(hash);
        if (tx != null) {
            return tx;
        }
        return null;
    }

    public synchronized void clearTransactions(int fromHeight) {
        if (fromHeight != 0) {
            throw new UnsupportedOperationException();
        }
        this.unspent.clear();
        this.spent.clear();
        this.pending.clear();
        this.inactive.clear();
        this.dead.clear();
        this.queueAutoSave();
    }

    synchronized EnumSet<WalletTransaction.Pool> getContainingPools(Transaction tx) {
        EnumSet<WalletTransaction.Pool> result = EnumSet.noneOf(WalletTransaction.Pool.class);
        Sha256Hash txHash = tx.getHash();
        if (this.unspent.containsKey(txHash)) {
            result.add(WalletTransaction.Pool.UNSPENT);
        }
        if (this.spent.containsKey(txHash)) {
            result.add(WalletTransaction.Pool.SPENT);
        }
        if (this.pending.containsKey(txHash)) {
            result.add(WalletTransaction.Pool.PENDING);
        }
        if (this.inactive.containsKey(txHash)) {
            result.add(WalletTransaction.Pool.INACTIVE);
        }
        if (this.dead.containsKey(txHash)) {
            result.add(WalletTransaction.Pool.DEAD);
        }
        return result;
    }

    synchronized int getPoolSize(WalletTransaction.Pool pool) {
        switch (pool) {
            case UNSPENT: {
                return this.unspent.size();
            }
            case SPENT: {
                return this.spent.size();
            }
            case PENDING: {
                return this.pending.size();
            }
            case INACTIVE: {
                return this.inactive.size();
            }
            case DEAD: {
                return this.dead.size();
            }
            case ALL: {
                return this.unspent.size() + this.spent.size() + this.pending.size() + this.inactive.size() + this.dead.size();
            }
        }
        throw new RuntimeException("Unreachable");
    }

    public synchronized Transaction createSend(Address address, BigInteger nanocoins) {
        SendRequest req = SendRequest.to(address, nanocoins);
        if (this.completeTx(req)) {
            return req.tx;
        }
        return null;
    }

    public synchronized Transaction sendCoinsOffline(SendRequest request) {
        try {
            if (!this.completeTx(request)) {
                return null;
            }
            this.commitTx(request.tx);
            return request.tx;
        }
        catch (VerificationException e) {
            throw new RuntimeException(e);
        }
    }

    public SendResult sendCoins(PeerGroup peerGroup, Address to, BigInteger value) {
        SendRequest request = SendRequest.to(to, value);
        return this.sendCoins(peerGroup, request);
    }

    public SendResult sendCoins(PeerGroup peerGroup, SendRequest request) {
        Transaction tx = this.sendCoinsOffline(request);
        if (tx == null) {
            return null;
        }
        SendResult result = new SendResult();
        result.tx = tx;
        result.broadcastComplete = peerGroup.broadcastTransaction(tx);
        return result;
    }

    public synchronized Transaction sendCoins(Peer peer, SendRequest request) throws IOException {
        Transaction tx = this.sendCoinsOffline(request);
        if (tx == null) {
            return null;
        }
        peer.sendMessage(tx);
        return tx;
    }

    public synchronized boolean completeTx(SendRequest req) {
        Preconditions.checkArgument(!req.completed, "Given SendRequest has already been completed.");
        BigInteger value = BigInteger.ZERO;
        for (TransactionOutput output : req.tx.getOutputs()) {
            value = value.add(output.getValue());
        }
        value = value.add(req.fee);
        log.info("Completing send tx with {} outputs totalling {}", req.tx.getOutputs().size(), (Object)Utils.bitcoinValueToFriendlyString(value));
        BigInteger valueGathered = BigInteger.ZERO;
        LinkedList<TransactionOutput> gathered = new LinkedList<TransactionOutput>();
        for (Transaction tx : this.unspent.values()) {
            if (!tx.isMature()) continue;
            for (TransactionOutput output : tx.getOutputs()) {
                if (!output.isAvailableForSpending() || !output.isMine(this)) continue;
                gathered.add(output);
                valueGathered = valueGathered.add(output.getValue());
            }
            if (valueGathered.compareTo(value) < 0) continue;
            break;
        }
        if (valueGathered.compareTo(value) < 0) {
            log.info("Insufficient value in wallet for send, missing " + Utils.bitcoinValueToFriendlyString(value.subtract(valueGathered)));
            return false;
        }
        Preconditions.checkState(gathered.size() > 0);
        req.tx.getConfidence().setConfidenceType(TransactionConfidence.ConfidenceType.NOT_SEEN_IN_CHAIN);
        BigInteger change = valueGathered.subtract(value);
        if (change.compareTo(BigInteger.ZERO) > 0) {
            Address changeAddress = req.changeAddress != null ? req.changeAddress : this.getChangeAddress();
            log.info("  with {} coins change", (Object)Utils.bitcoinValueToFriendlyString(change));
            req.tx.addOutput(new TransactionOutput(this.params, req.tx, change, changeAddress));
        }
        for (TransactionOutput output : gathered) {
            req.tx.addInput(output);
        }
        try {
            req.tx.signInputs(Transaction.SigHash.ALL, this);
        }
        catch (ScriptException e) {
            throw new RuntimeException(e);
        }
        req.completed = true;
        log.info("  completed {}", (Object)req.tx.getHashAsString());
        return true;
    }

    synchronized Address getChangeAddress() {
        Preconditions.checkState(this.keychain.size() > 0, "Can't send value without an address to use for receiving change");
        ECKey first = this.keychain.get(0);
        return first.toAddress(this.params);
    }

    public synchronized void addKey(final ECKey key) {
        Preconditions.checkArgument(!this.keychain.contains(key), "Key already present");
        this.keychain.add(key);
        EventListenerInvoker.invoke(this.eventListeners, new EventListenerInvoker<WalletEventListener>(){

            @Override
            public void invoke(WalletEventListener listener) {
                listener.onKeyAdded(key);
            }
        });
        if (this.autosaveToFile != null) {
            this.autoSave();
        }
    }

    public synchronized ECKey findKeyFromPubHash(byte[] pubkeyHash) {
        for (ECKey key : this.keychain) {
            if (!Arrays.equals(key.getPubKeyHash(), pubkeyHash)) continue;
            return key;
        }
        return null;
    }

    public synchronized boolean isPubKeyHashMine(byte[] pubkeyHash) {
        return this.findKeyFromPubHash(pubkeyHash) != null;
    }

    public synchronized ECKey findKeyFromPubKey(byte[] pubkey) {
        for (ECKey key : this.keychain) {
            if (!Arrays.equals(key.getPubKey(), pubkey)) continue;
            return key;
        }
        return null;
    }

    public synchronized boolean isPubKeyMine(byte[] pubkey) {
        return this.findKeyFromPubKey(pubkey) != null;
    }

    public synchronized BigInteger getBalance() {
        return this.getBalance(BalanceType.AVAILABLE);
    }

    public synchronized BigInteger getBalance(BalanceType balanceType) {
        BigInteger available = BigInteger.ZERO;
        for (Transaction tx : this.unspent.values()) {
            if (balanceType == BalanceType.AVAILABLE && !tx.isMature()) continue;
            for (TransactionOutput output : tx.getOutputs()) {
                if (!output.isMine(this) || !output.isAvailableForSpending()) continue;
                available = available.add(output.getValue());
            }
        }
        if (balanceType == BalanceType.AVAILABLE) {
            return available;
        }
        Preconditions.checkState(balanceType == BalanceType.ESTIMATED);
        BigInteger estimated = available;
        for (Transaction tx : this.pending.values()) {
            for (TransactionOutput output : tx.getOutputs()) {
                if (!output.isMine(this)) continue;
                estimated = estimated.add(output.getValue());
            }
        }
        return estimated;
    }

    public synchronized String toString() {
        return this.toString(false);
    }

    public synchronized String toString(boolean includePrivateKeys) {
        StringBuilder builder = new StringBuilder();
        builder.append(String.format("Wallet containing %s BTC in:%n", Utils.bitcoinValueToFriendlyString(this.getBalance())));
        builder.append(String.format("  %d unspent transactions%n", this.unspent.size()));
        builder.append(String.format("  %d spent transactions%n", this.spent.size()));
        builder.append(String.format("  %d pending transactions%n", this.pending.size()));
        builder.append(String.format("  %d inactive transactions%n", this.inactive.size()));
        builder.append(String.format("  %d dead transactions%n", this.dead.size()));
        builder.append(String.format("Last seen best block: %s%n", this.getLastBlockSeenHash()));
        builder.append("\nKeys:\n");
        for (ECKey key : this.keychain) {
            builder.append("  addr:");
            builder.append(key.toAddress(this.params));
            builder.append(" ");
            builder.append(includePrivateKeys ? key.toStringWithPrivate() : key.toString());
            builder.append("\n");
        }
        if (this.unspent.size() > 0) {
            builder.append("\nUNSPENT:\n");
            this.toStringHelper(builder, this.unspent);
        }
        if (this.spent.size() > 0) {
            builder.append("\nSPENT:\n");
            this.toStringHelper(builder, this.spent);
        }
        if (this.pending.size() > 0) {
            builder.append("\nPENDING:\n");
            this.toStringHelper(builder, this.pending);
        }
        if (this.inactive.size() > 0) {
            builder.append("\nINACTIVE:\n");
            this.toStringHelper(builder, this.inactive);
        }
        if (this.dead.size() > 0) {
            builder.append("\nDEAD:\n");
            this.toStringHelper(builder, this.dead);
        }
        return builder.toString();
    }

    private void toStringHelper(StringBuilder builder, Map<Sha256Hash, Transaction> transactionMap) {
        for (Transaction tx : transactionMap.values()) {
            try {
                builder.append("Sends ");
                builder.append(Utils.bitcoinValueToFriendlyString(tx.getValueSentFromMe(this)));
                builder.append(" and receives ");
                builder.append(Utils.bitcoinValueToFriendlyString(tx.getValueSentToMe(this)));
                builder.append(", total value ");
                builder.append(Utils.bitcoinValueToFriendlyString(tx.getValue(this)));
                builder.append(".\n");
            }
            catch (ScriptException e) {
                // empty catch block
            }
            builder.append(tx);
        }
    }

    synchronized void reorganize(StoredBlock splitPoint, List<StoredBlock> oldBlocks, List<StoredBlock> newBlocks) throws VerificationException {
        ArrayList<Sha256Hash> oldBlockHashes = new ArrayList<Sha256Hash>(oldBlocks.size());
        ArrayList<Sha256Hash> newBlockHashes = new ArrayList<Sha256Hash>(newBlocks.size());
        log.info("Old part of chain (top to bottom):");
        for (StoredBlock b : oldBlocks) {
            log.info("  {}", (Object)b.getHeader().getHashAsString());
            oldBlockHashes.add(b.getHeader().getHash());
        }
        log.info("New part of chain (top to bottom):");
        for (StoredBlock b : newBlocks) {
            log.info("  {}", (Object)b.getHeader().getHashAsString());
            newBlockHashes.add(b.getHeader().getHash());
        }
        HashMap<Sha256Hash, Transaction> oldChainTransactions = new HashMap<Sha256Hash, Transaction>();
        HashMap<Sha256Hash, Transaction> onlyOldChainTransactions = new HashMap<Sha256Hash, Transaction>();
        HashMap<Sha256Hash, Transaction> newChainTransactions = new HashMap<Sha256Hash, Transaction>();
        HashMap<Sha256Hash, Transaction> commonChainTransactions = new HashMap<Sha256Hash, Transaction>();
        HashMap<Sha256Hash, Transaction> all = new HashMap<Sha256Hash, Transaction>();
        all.putAll(this.unspent);
        all.putAll(this.spent);
        all.putAll(this.inactive);
        for (Transaction tx : this.dead.values()) {
            if (!tx.isCoinBase()) continue;
            all.put(tx.getHash(), tx);
        }
        for (Transaction tx : all.values()) {
            boolean alreadyPresent;
            boolean inCommonSection;
            Collection<Sha256Hash> appearsIn = tx.getAppearsInHashes();
            Preconditions.checkNotNull(appearsIn);
            boolean inOldSection = !Collections.disjoint(appearsIn, oldBlockHashes);
            boolean inNewSection = !Collections.disjoint(appearsIn, newBlockHashes);
            boolean bl = inCommonSection = !inNewSection && !inOldSection;
            if (inCommonSection) {
                alreadyPresent = commonChainTransactions.put(tx.getHash(), tx) != null;
                Preconditions.checkState(!alreadyPresent, "Transaction appears twice in common chain segment");
                continue;
            }
            if (inOldSection) {
                alreadyPresent = oldChainTransactions.put(tx.getHash(), tx) != null;
                Preconditions.checkState(!alreadyPresent, "Transaction appears twice in old chain segment");
                if (!inNewSection) {
                    alreadyPresent = onlyOldChainTransactions.put(tx.getHash(), tx) != null;
                    Preconditions.checkState(!alreadyPresent, "Transaction appears twice in only-old map");
                }
            }
            if (!inNewSection) continue;
            alreadyPresent = newChainTransactions.put(tx.getHash(), tx) != null;
            Preconditions.checkState(!alreadyPresent, "Transaction appears twice in new chain segment");
        }
        boolean affectedUs = !((Object)oldChainTransactions).equals(newChainTransactions);
        log.info(affectedUs ? "Re-org affected our transactions" : "Re-org had no effect on our transactions");
        if (!affectedUs) {
            return;
        }
        for (Transaction tx : onlyOldChainTransactions.values()) {
            log.info("  Only Old: {}", (Object)tx.getHashAsString());
        }
        for (Transaction tx : oldChainTransactions.values()) {
            log.info("  Old: {}", (Object)tx.getHashAsString());
        }
        for (Transaction tx : newChainTransactions.values()) {
            log.info("  New: {}", (Object)tx.getHashAsString());
        }
        for (Transaction tx : all.values()) {
            tx.disconnectInputs();
        }
        for (Transaction tx : this.pending.values()) {
            tx.disconnectInputs();
        }
        for (Transaction tx : commonChainTransactions.values()) {
            TransactionInput badInput = tx.connectForReorganize(all);
            Preconditions.checkState(badInput == null, "Failed to connect %s, %s", tx.getHashAsString(), badInput == null ? "" : badInput.toString());
        }
        log.info("Moving transactions");
        this.unspent.clear();
        this.spent.clear();
        this.inactive.clear();
        for (Transaction tx : commonChainTransactions.values()) {
            int unspentOutputs = 0;
            for (TransactionOutput output : tx.getOutputs()) {
                if (!output.isAvailableForSpending() || !output.isMine(this)) continue;
                ++unspentOutputs;
            }
            if (unspentOutputs > 0) {
                log.info("  TX {} ->unspent", (Object)tx.getHashAsString());
                this.unspent.put(tx.getHash(), tx);
                continue;
            }
            log.info("  TX {} ->spent", (Object)tx.getHashAsString());
            this.spent.put(tx.getHash(), tx);
        }
        for (Transaction tx : onlyOldChainTransactions.values()) {
            tx.notifyNotOnBestChain();
            if (!tx.isCoinBase()) continue;
            if (this.unspent.containsKey(tx.getHash())) {
                log.info("  coinbase tx {} unspent->dead", (Object)tx.getHashAsString());
                this.unspent.remove(tx.getHash());
            } else if (this.spent.containsKey(tx.getHash())) {
                log.info("  coinbase tx {} spent->dead", (Object)tx.getHashAsString());
                this.spent.remove(tx.getHash());
            }
            this.dead.put(tx.getHash(), tx);
            tx.getConfidence().setConfidenceType(TransactionConfidence.ConfidenceType.DEAD);
        }
        Collections.reverse(newBlocks);
        int depthToSubtract = oldBlocks.size();
        BigInteger workDoneToSubtract = BigInteger.ZERO;
        for (StoredBlock b : oldBlocks) {
            workDoneToSubtract = workDoneToSubtract.add(b.getHeader().getWork());
        }
        log.info("DepthToSubtract = " + depthToSubtract + ", workDoneToSubtract = " + workDoneToSubtract);
        this.subtractDepthAndWorkDone(depthToSubtract, workDoneToSubtract, this.spent.values());
        this.subtractDepthAndWorkDone(depthToSubtract, workDoneToSubtract, this.unspent.values());
        this.subtractDepthAndWorkDone(depthToSubtract, workDoneToSubtract, this.dead.values());
        this.setLastBlockSeenHash(splitPoint.getHeader().getHash());
        for (StoredBlock b : newBlocks) {
            log.info("Replaying block {}", (Object)b.getHeader().getHashAsString());
            HashSet<Transaction> txns = new HashSet<Transaction>();
            Sha256Hash blockHash = b.getHeader().getHash();
            for (Transaction tx : newChainTransactions.values()) {
                if (!tx.getAppearsInHashes().contains(blockHash)) continue;
                txns.add(tx);
                log.info("  containing tx {}", (Object)tx.getHashAsString());
            }
            if (!txns.isEmpty()) {
                for (Transaction t : txns) {
                    try {
                        this.receive(t, b, BlockChain.NewBlockType.BEST_CHAIN, true);
                    }
                    catch (ScriptException e) {
                        throw new RuntimeException(e);
                    }
                }
            }
            this.notifyNewBestBlock(b.getHeader());
        }
        HashMap<Sha256Hash, Transaction> pool = new HashMap<Sha256Hash, Transaction>();
        pool.putAll(this.unspent);
        pool.putAll(this.spent);
        pool.putAll(this.pending);
        HashMap<Sha256Hash, Transaction> toReprocess = new HashMap<Sha256Hash, Transaction>();
        toReprocess.putAll(onlyOldChainTransactions);
        toReprocess.putAll(this.pending);
        log.info("Reprocessing transactions not in new best chain:");
        for (Transaction tx : this.dead.values()) {
            this.reprocessUnincludedTxAfterReorg(pool, tx);
        }
        for (Transaction tx : toReprocess.values()) {
            this.reprocessUnincludedTxAfterReorg(pool, tx);
        }
        log.info("post-reorg balance is {}", (Object)Utils.bitcoinValueToFriendlyString(this.getBalance()));
        EventListenerInvoker.invoke(this.eventListeners, new EventListenerInvoker<WalletEventListener>(){

            @Override
            public void invoke(WalletEventListener listener) {
                listener.onReorganize(Wallet.this);
            }
        });
        Preconditions.checkState(this.isConsistent());
    }

    private synchronized void subtractDepthAndWorkDone(int depthToSubtract, BigInteger workDoneToSubtract, Collection<Transaction> transactions) {
        for (Transaction tx : transactions) {
            if (tx.getConfidence().getConfidenceType() != TransactionConfidence.ConfidenceType.BUILDING) continue;
            tx.getConfidence().setDepthInBlocks(tx.getConfidence().getDepthInBlocks() - depthToSubtract);
            tx.getConfidence().setWorkDone(tx.getConfidence().getWorkDone().subtract(workDoneToSubtract));
        }
    }

    private void reprocessUnincludedTxAfterReorg(Map<Sha256Hash, Transaction> pool, Transaction tx) {
        boolean isDeadCoinbase;
        log.info("TX {}", (Object)(tx.getHashAsString() + ", confidence = " + tx.getConfidence().getConfidenceType().name()));
        boolean bl = isDeadCoinbase = tx.isCoinBase() && TransactionConfidence.ConfidenceType.DEAD == tx.getConfidence().getConfidenceType();
        if (isDeadCoinbase) {
            return;
        }
        int numInputs = tx.getInputs().size();
        int noSuchTx = 0;
        int success = 0;
        boolean isDead = false;
        HashSet<Transaction> connectedTransactions = new HashSet<Transaction>();
        for (TransactionInput input : tx.getInputs()) {
            TransactionInput.ConnectionResult result = input.connect(pool, TransactionInput.ConnectMode.ABORT_ON_CONFLICT);
            if (result == TransactionInput.ConnectionResult.SUCCESS) {
                ++success;
                TransactionOutput connectedOutput = Preconditions.checkNotNull(input.getConnectedOutput(pool));
                connectedTransactions.add(Preconditions.checkNotNull(connectedOutput.parentTransaction));
                continue;
            }
            if (result == TransactionInput.ConnectionResult.NO_SUCH_TX) {
                ++noSuchTx;
                continue;
            }
            if (result != TransactionInput.ConnectionResult.ALREADY_SPENT) continue;
            isDead = true;
            log.info("   ->dead, will not confirm now unless there's another re-org", (Object)tx.getHashAsString());
            TransactionOutput doubleSpent = input.getConnectedOutput(pool);
            Transaction replacement = doubleSpent.getSpentBy().getParentTransaction();
            this.dead.put(tx.getHash(), tx);
            this.pending.remove(tx.getHash());
            tx.getConfidence().setOverridingTransaction(replacement);
            break;
        }
        if (isDead) {
            return;
        }
        if (noSuchTx == numInputs) {
            log.info("   ->inactive", (Object)(tx.getHashAsString() + ", confidence = " + tx.getConfidence().getConfidenceType().name()));
            this.inactive.put(tx.getHash(), tx);
            this.dead.remove(tx.getHash());
        } else if (success == numInputs - noSuchTx) {
            log.info("   ->pending", (Object)(tx.getHashAsString() + ", confidence = " + tx.getConfidence().getConfidenceType().name()));
            this.pending.put(tx.getHash(), tx);
            this.dead.remove(tx.getHash());
        }
        for (Transaction maybeSpent : connectedTransactions) {
            this.maybeMoveTxToSpent(maybeSpent, "reorg");
        }
    }

    private void invokeOnTransactionConfidenceChanged(final Transaction tx) {
        EventListenerInvoker.invoke(this.eventListeners, new EventListenerInvoker<WalletEventListener>(){

            @Override
            public void invoke(WalletEventListener listener) {
                listener.onTransactionConfidenceChanged(Wallet.this, tx);
            }
        });
    }

    public synchronized Collection<Transaction> getPendingTransactions() {
        return Collections.unmodifiableCollection(this.pending.values());
    }

    public synchronized long getEarliestKeyCreationTime() {
        if (this.keychain.size() == 0) {
            return Utils.now().getTime() / 1000L;
        }
        long earliestTime = Long.MAX_VALUE;
        for (ECKey key : this.keychain) {
            earliestTime = Math.min(key.getCreationTimeSeconds(), earliestTime);
        }
        return earliestTime;
    }

    public synchronized PeerEventListener getPeerEventListener() {
        if (this.peerEventListener == null) {
            this.peerEventListener = new AbstractPeerEventListener(){

                public void onTransaction(Peer peer, Transaction t) {
                    try {
                        Wallet.this.receivePending(t);
                    }
                    catch (VerificationException e) {
                        log.warn("Received broadcast transaction that does not validate: {}", t);
                        log.warn("VerificationException caught", e);
                    }
                    catch (ScriptException e) {
                        log.warn("Received broadcast transaction with not understood scripts: {}", t);
                        log.warn("ScriptException caught", e);
                    }
                }
            };
        }
        return this.peerEventListener;
    }

    public Sha256Hash getLastBlockSeenHash() {
        return this.lastBlockSeenHash;
    }

    public void setLastBlockSeenHash(Sha256Hash lastBlockSeenHash) {
        this.lastBlockSeenHash = lastBlockSeenHash;
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    public static enum BalanceType {
        ESTIMATED,
        AVAILABLE;

    }

    public static class SendRequest {
        public Transaction tx;
        public Address changeAddress;
        public BigInteger fee = BigInteger.ZERO;
        private boolean completed;

        private SendRequest() {
        }

        public static SendRequest to(Address destination, BigInteger value) {
            SendRequest req = new SendRequest();
            req.tx = new Transaction(destination.getParameters());
            req.tx.addOutput(value, destination);
            return req;
        }

        public static SendRequest to(NetworkParameters params, ECKey destination, BigInteger value) {
            SendRequest req = new SendRequest();
            req.tx = new Transaction(params);
            req.tx.addOutput(value, destination);
            return req;
        }

        public static SendRequest forTx(Transaction tx) {
            SendRequest req = new SendRequest();
            req.tx = tx;
            return req;
        }
    }

    public static class SendResult {
        public Transaction tx;
        public ListenableFuture<Transaction> broadcastComplete;
    }

    public static interface AutosaveEventListener {
        public boolean caughtException(Throwable var1);

        public void onBeforeAutoSave(File var1);

        public void onAfterAutoSave(File var1);
    }

    private static class AutosaveThread
    extends Thread {
        private static DelayQueue<WalletSaveRequest> walletRefs = new DelayQueue();
        private static AutosaveThread globalThread;

        private AutosaveThread() {
            this.setDaemon(true);
            this.setName("Wallet auto save thread");
            this.setPriority(1);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public static void maybeStart() {
            if (walletRefs.size() == 0) {
                return;
            }
            Class<AutosaveThread> clazz = AutosaveThread.class;
            synchronized (AutosaveThread.class) {
                if (globalThread == null) {
                    globalThread = new AutosaveThread();
                    globalThread.start();
                }
                // ** MonitorExit[var0] (shouldn't be in output)
                return;
            }
        }

        public static void registerForSave(Wallet wallet, long delayMsec) {
            walletRefs.add(new WalletSaveRequest(wallet, delayMsec));
            AutosaveThread.maybeStart();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        public void run() {
            log.info("Auto-save thread starting up");
            try {
                while (true) {
                    WalletSaveRequest req;
                    if ((req = (WalletSaveRequest)walletRefs.poll(5L, TimeUnit.SECONDS)) == null) {
                        if (walletRefs.size() != 0) continue;
                        break;
                    }
                    Wallet wallet = req.wallet;
                    synchronized (wallet) {
                        if (req.wallet.dirty && req.wallet.autoSave()) {
                            break;
                        }
                    }
                }
            }
            catch (InterruptedException e) {
                log.error("Auto-save thread interrupted during wait", e);
            }
            log.info("Auto-save thread shutting down");
            Class<AutosaveThread> clazz = AutosaveThread.class;
            synchronized (AutosaveThread.class) {
                Preconditions.checkState(globalThread == this);
                globalThread = null;
                // ** MonitorExit[var1_1] (shouldn't be in output)
                AutosaveThread.maybeStart();
                return;
            }
        }

        private static class WalletSaveRequest
        implements Delayed {
            public final Wallet wallet;
            public final long startTimeMs = System.currentTimeMillis();
            public final long requestedDelayMs;

            public WalletSaveRequest(Wallet wallet, long requestedDelayMs) {
                this.requestedDelayMs = requestedDelayMs;
                this.wallet = wallet;
            }

            public long getDelay(TimeUnit timeUnit) {
                long delayRemainingMs = this.requestedDelayMs - (System.currentTimeMillis() - this.startTimeMs);
                return timeUnit.convert(delayRemainingMs, TimeUnit.MILLISECONDS);
            }

            public int compareTo(Delayed delayed) {
                if (delayed == this) {
                    return 0;
                }
                long delta = this.getDelay(TimeUnit.MILLISECONDS) - delayed.getDelay(TimeUnit.MILLISECONDS);
                return delta > 0L ? 1 : (delta < 0L ? -1 : 0);
            }
        }
    }
}

