/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.uri;

import com.google.bitcoin.core.Address;
import com.google.bitcoin.core.AddressFormatException;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.Utils;
import com.google.bitcoin.uri.BitcoinURIParseException;
import com.google.bitcoin.uri.OptionalFieldValidationException;
import com.google.bitcoin.uri.RequiredFieldValidationException;
import com.google.common.base.Preconditions;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.LinkedHashMap;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BitcoinURI {
    private static final Logger log = LoggerFactory.getLogger(BitcoinURI.class);
    public static final String FIELD_MESSAGE = "message";
    public static final String FIELD_LABEL = "label";
    public static final String FIELD_AMOUNT = "amount";
    public static final String FIELD_ADDRESS = "address";
    public static final String BITCOIN_SCHEME = "bitcoin";
    private static final String ENCODED_SPACE_CHARACTER = "%20";
    private static final String AMPERSAND_SEPARATOR = "&";
    private static final String QUESTION_MARK_SEPARATOR = "?";
    private final Map<String, Object> parameterMap = new LinkedHashMap<String, Object>();

    public BitcoinURI(String uri) {
        this(null, uri);
    }

    public BitcoinURI(NetworkParameters params, String input) {
        String[] nameValuePairTokens;
        String schemeSpecificPart;
        URI uri;
        Preconditions.checkNotNull(input);
        log.debug("Attempting to parse '{}' for {}", (Object)input, (Object)(params == null ? "any" : params.getId()));
        try {
            uri = new URI(input);
        }
        catch (URISyntaxException e) {
            throw new BitcoinURIParseException("Bad URI syntax", e);
        }
        if (input.startsWith("bitcoin://")) {
            schemeSpecificPart = input.substring("bitcoin://".length());
        } else if (input.startsWith("bitcoin:")) {
            schemeSpecificPart = input.substring("bitcoin:".length());
        } else {
            throw new BitcoinURIParseException("Unsupported URI scheme: " + uri.getScheme());
        }
        String[] addressSplitTokens = schemeSpecificPart.split("\\?");
        if (addressSplitTokens.length == 0 || "".equals(addressSplitTokens[0])) {
            throw new BitcoinURIParseException("Missing address");
        }
        String addressToken = addressSplitTokens[0];
        if (addressSplitTokens.length == 1) {
            nameValuePairTokens = new String[]{};
        } else if (addressSplitTokens.length == 2) {
            nameValuePairTokens = addressSplitTokens[1].split(AMPERSAND_SEPARATOR);
        } else {
            throw new BitcoinURIParseException("Too many question marks in URI '" + uri + "'");
        }
        this.parseParameters(params, addressToken, nameValuePairTokens);
    }

    private void parseParameters(NetworkParameters params, String addressToken, String[] nameValuePairTokens) {
        try {
            Address address = new Address(params, addressToken);
            this.putWithValidation(FIELD_ADDRESS, address);
        }
        catch (AddressFormatException e) {
            throw new BitcoinURIParseException("Bad address", e);
        }
        for (int i = 0; i < nameValuePairTokens.length; ++i) {
            String[] tokens = nameValuePairTokens[i].split("=");
            if (tokens.length != 2 || "".equals(tokens[0])) {
                throw new BitcoinURIParseException("Malformed Bitcoin URI - cannot parse name value pair '" + nameValuePairTokens[i] + "'");
            }
            String nameToken = tokens[0].toLowerCase();
            String valueToken = tokens[1];
            if (FIELD_AMOUNT.equals(nameToken)) {
                try {
                    BigInteger amount = Utils.toNanoCoins(valueToken);
                    this.putWithValidation(FIELD_AMOUNT, amount);
                    continue;
                }
                catch (NumberFormatException e) {
                    throw new OptionalFieldValidationException("'" + valueToken + "' value is not a valid amount", e);
                }
            }
            if (nameToken.startsWith("req-")) {
                throw new RequiredFieldValidationException("'" + nameToken + "' is required but not known, this URI is not valid");
            }
            try {
                this.putWithValidation(nameToken, URLDecoder.decode(valueToken, "UTF-8"));
                continue;
            }
            catch (UnsupportedEncodingException e) {
                throw new RuntimeException(e);
            }
        }
    }

    private void putWithValidation(String key, Object value) {
        if (this.parameterMap.containsKey(key)) {
            throw new BitcoinURIParseException("'" + key + "' is duplicated, URI is invalid");
        }
        this.parameterMap.put(key, value);
    }

    public Address getAddress() {
        return (Address)this.parameterMap.get(FIELD_ADDRESS);
    }

    public BigInteger getAmount() {
        return (BigInteger)this.parameterMap.get(FIELD_AMOUNT);
    }

    public String getLabel() {
        return (String)this.parameterMap.get(FIELD_LABEL);
    }

    public String getMessage() {
        return (String)this.parameterMap.get(FIELD_MESSAGE);
    }

    public Object getParameterByName(String name) {
        return this.parameterMap.get(name);
    }

    public String toString() {
        StringBuilder builder = new StringBuilder("BitcoinURI[");
        boolean first = true;
        for (Map.Entry<String, Object> entry : this.parameterMap.entrySet()) {
            if (first) {
                first = false;
            } else {
                builder.append(",");
            }
            builder.append("'").append(entry.getKey()).append("'=").append("'").append(entry.getValue().toString()).append("'");
        }
        builder.append("]");
        return builder.toString();
    }

    public static String convertToBitcoinURI(Address address, BigInteger amount, String label, String message) {
        return BitcoinURI.convertToBitcoinURI(address.toString(), amount, label, message);
    }

    public static String convertToBitcoinURI(String address, BigInteger amount, String label, String message) {
        Preconditions.checkNotNull(address);
        if (amount != null && amount.compareTo(BigInteger.ZERO) < 0) {
            throw new IllegalArgumentException("Amount must be positive");
        }
        StringBuilder builder = new StringBuilder();
        builder.append(BITCOIN_SCHEME).append(":").append(address);
        boolean questionMarkHasBeenOutput = false;
        if (amount != null) {
            builder.append(QUESTION_MARK_SEPARATOR).append(FIELD_AMOUNT).append("=");
            builder.append(Utils.bitcoinValueToPlainString(amount));
            questionMarkHasBeenOutput = true;
        }
        if (label != null && !"".equals(label)) {
            if (questionMarkHasBeenOutput) {
                builder.append(AMPERSAND_SEPARATOR);
            } else {
                builder.append(QUESTION_MARK_SEPARATOR);
                questionMarkHasBeenOutput = true;
            }
            builder.append(FIELD_LABEL).append("=").append(BitcoinURI.encodeURLString(label));
        }
        if (message != null && !"".equals(message)) {
            if (questionMarkHasBeenOutput) {
                builder.append(AMPERSAND_SEPARATOR);
            } else {
                builder.append(QUESTION_MARK_SEPARATOR);
                questionMarkHasBeenOutput = true;
            }
            builder.append(FIELD_MESSAGE).append("=").append(BitcoinURI.encodeURLString(message));
        }
        return builder.toString();
    }

    static String encodeURLString(String stringToEncode) {
        try {
            return URLEncoder.encode(stringToEncode, "UTF-8").replace("+", ENCODED_SPACE_CHARACTER);
        }
        catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }
}

